<?php

namespace App\Services;

use App\Http\Resources\WaytopayByCompanyResource;
use App\Models\PaymentCuotas;
use App\Models\PaymentTools;
use App\Models\WayToPay;
use App\Models\WayToPayForCompany;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

class WayToPayService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $wayToPay = WayToPay::where('status', 1)->orderBy('id', 'asc');

        $search = isset($params['search']) ? $params['search'] : '';
        if ($search) {
            $wayToPay->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $wayToPay = !empty($withPagination)
            ? $wayToPay->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $wayToPay->get();

        return $this->successResponse('Lectura exitosa.', $wayToPay);
    }

    public function listByCompany($params, $withPagination)
    {
        $details = WayToPayForCompany::active()
            ->company()->orderBy('id', 'asc');

        $details = !empty($withPagination)
            ? $details->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $details->get();

        $details = WaytopayByCompanyResource::collection($details->load('company', 'wayToPay'));

        return $this->successResponse('Lectura exitosa.', $details);
    }

    public function listPaymentCuotas($params, $withPagination)
    {
        $cuotas = PaymentCuotas::where('status', 1)->orderBy('id', 'asc');

        $cuotas = !empty($withPagination)
            ? $cuotas->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $cuotas->get();

        return $this->successResponse('Lectura exitosa.', $cuotas);
    }

    public function listPaymentTools($params, $withPagination)
    {
        $tools = PaymentTools::where('status', 1)->orderBy('id', 'desc');

        $search = isset($params['search']) ? $params['search'] : '';
        if ($search) {
            $tools->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $tools = !empty($withPagination)
            ? $tools->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $tools->get();

        return $this->successResponse('Lectura exitosa.', $tools);
    }

    public function assignPaymentMethod($params)
    {
        DB::beginTransaction();
        try {
            $existingAssignment  = WayToPayForCompany::where('idway_to_pay', $params['idway_to_pay'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($existingAssignment) return $this->errorResponse('La forma de pago ya ha sido asignada a la compañia actual.', 400);

            $assignment = WayToPayForCompany::create([
                'idway_to_pay' => $params['idway_to_pay'],
                'idtools' => $params['idtools'],
                'cuotas' => $params['cuotas'],
                'idcuotas' => $params['cuotas'] == '1' ? $params['idcuotas'] : null,
                'idcompany' => isset($params['idcompany']) ? $params['idcompany'] : null,
            ]);
            $assignment->save();

            DB::commit();
            return $this->successResponse('Asignación de pago creada satisfactoriamente.', $assignment);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la asignación de pago.', $th->getMessage());
        }
    }

    public function updatePaymentAssignment($idwaytopay, $params)
    {
        DB::beginTransaction();
        try {

            $existingWaytopay = WayToPay::where('id', $idwaytopay)->where('status', 1)->first();
            if (!$existingWaytopay) return $this->errorResponse('No se encontró la forma de pago.', 400);

            $assignment  = WayToPayForCompany::where('idway_to_pay', $idwaytopay)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if (!$assignment) return $this->errorResponse('La forma de pago no se encuentra asignada a la compañia.', 400);

            $assignment->idtools = $params['idtools'];
            $assignment->cuotas = $params['cuotas'];
            $assignment->idcuotas = $params['cuotas'] == '1' ? $params['idcuotas'] : null;
            $assignment->save();

            DB::commit();
            return $this->successResponse('Asignación de pago actualizada satisfactoriamente.', $assignment);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la asignación de pago.', $th->getMessage());
        }
    }
}
