<?php

namespace App\Services;

use App\Http\Resources\DetailUserCompanyResource;
use App\Http\Resources\UserResource;
use App\Mail\ResendCredentialsMail;
use App\Mail\WelcomeUserMail;
use App\Models\DetailUserCompany;
use App\Models\Privilege;
use App\Models\User;
use App\Models\UserType;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Str;

/**
 * Class UserService
 * @package App\Services
 */
class UserService
{
    use HasResponse;

    public function list($params, $withPagination)
    {

        $details = DetailUserCompany::where('status', 1)->orderBy('id', 'desc');

        $details = !empty($withPagination)
            ? $details->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $details->get();

        $details = DetailUserCompanyResource::collection($details->load('user', 'userType', 'privilege', 'company'));
        $details = $details->toResponse(request())->getData()->data;

        if(sizeof($details) > 0){
            foreach ($details as $detail) {
                $idcompanies = DetailUserCompany::where('iduser', $detail->id)->where('status', 1)->pluck('idcompany')->unique()->values()->all();
                $idcompanies = json_encode($idcompanies);
                $detail->idcompanies = '' . $idcompanies . '';
                $detail->idcompany = null;
            }
        }

        $search = isset($params['search']) ? $params['search'] : '';
        $users = collect($details)->filter(function ($item) use ($search) {
            return stripos($item->name, $search) !== false ||
                stripos($item->username, $search) !== false ||
                stripos($item->rut, $search) !== false ||
                stripos($item->email, $search) !== false ||
                stripos($item->first_surname, $search) !== false;
        })->values()->all();

        return $this->successResponse('OK', $users);
    }

    public function listByCompany($params, $withPagination)
    {

        $users = DetailUserCompany::where('status', 1)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $users->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $users = !empty($withPagination)
            ? $users->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $users->get();

        $users = DetailUserCompanyResource::collection($users->load('user', 'userType', 'privilege', 'company'));
        $users = $users->toResponse(request())->getData()->data;

        $search = isset($params['search']) ? $params['search'] : '';
        $users = collect($users)->filter(function ($item) use ($search) {
            return stripos($item->name, $search) !== false ||
                stripos($item->username, $search) !== false ||
                stripos($item->rut, $search) !== false ||
                stripos($item->email, $search) !== false ||
                stripos($item->first_surname, $search) !== false;
        })->values()->all();

        return $this->successResponse('Lectura exitosa.', $users);
    }

    public function listUserType()
    {
        // $userType = UserType::userTypeFilters();
        $userType = UserType::typeUserFilters();

        //omitir superadmin
        $userType = $userType->where('id', '!=', 1);

        return $this->successResponse('OK', $userType->get());
    }

    public function listPrivilege()
    {
        $privileges = Privilege::where('status', 1);

        return $this->successResponse('OK', $privileges->get());
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            $usernameExisting = DetailUserCompany::whereHas('company', function ($query) use ($params) {
                return $query->where('id', $params['idcompany']);
            })->whereHas('user', function ($query) use ($params) {
                    return $query->where('username', $params['username']);
                })->where('status', 1)->first();

            if($usernameExisting) return $this->errorResponse('El nombre de usuario ya se encuentra registrado.', 400);

            $rutExisting= User::where('rut', $params['rut'])->where('status', 1)->first();
            if($rutExisting) return $this->errorResponse('El rut ya se encuentra registrado.', 400);

            if($params['iduser_type'] == '1') return $this->errorResponse('Ud. no tiene permiso para registrar este tipo de usuario.', 400);

            // $password = Str::random(10);
            $password = 'demo';
            $user = User::create([
                'rut'         => $params['rut'],
                'email'       => $params['email'],
                'name'        => $params['name'],
                'surname'     => $params['first_surname'].' '. $params['second_surname'],
                'gender'      => $params['gender'],
                'type_person' => $params['type_person'],
                'age'         => $params['age'],
                'birthdate'   => $params['birthdate'],
                'idcompany'   => $params['idcompany'],
                'address'     => $params['address'] ?? null,
                'username'    => $params['username'],
                'password'    => bcrypt($password),
                'encrypted_password' => Crypt::encryptString($password)
            ]);
            $user->fresh();

            $params['iduser'] = $user->id;
            $this->registerDetailUser($params);

            // $mail = $this->sendMail($params);
            // if(!$mail->original['status']) return $mail;

            DB::commit();
            return $this->successResponse('Usuario creado satisfactoriamente.', $user);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de usuario.', $th->getMessage());
        }
    }

    public function registerBySuperadmin($params)
    {
        DB::beginTransaction();
        try {

            $idcompanies = json_decode($params['idcompanies'], true);

            #validar si el nombre de usurio existe almenos en una compañia
            $usernameExisting = DetailUserCompany::whereHas('company', function ($query) use ($idcompanies) {
                return $query->whereIn('id', $idcompanies);
            })->whereHas('user', function ($query) use ($params) {
                    return $query->where('username', $params['username']);
                })->where('status', 1)->first();
            if($usernameExisting) return $this->errorResponse('El nombre de usuario ya se encuentra registrado.', 400);

            $rutExisting= User::where('rut', $params['rut'])->where('status', 1)->first();
            if($rutExisting) return $this->errorResponse('El rut ya se encuentra registrado.', 400);

            $emailExisting= User::where('email', $params['email'])->where('status', 1)->first();
            if($emailExisting) return $this->errorResponse('El email ya se encuentra registrado.', 400);

            $user = User::create([
                'rut' => $params['rut'],
                'username' => $params['username'],
                'name' => $params['name'],
                'first_surname' => $params['first_surname'],
                'second_surname' => $params['second_surname'],
                'gender' => $params['gender'],
                'type_person' => $params['type_person'],
                'age' => $params['age'],
                'birthdate' => $params['birthdate'],
                'email' => $params['email'],
                'phone' => $params['phone'],
                'password' => bcrypt($params['password']),
                'encrypted_password' => Crypt::encryptString($params['password']),
            ]);
            $user->fresh();

            if(sizeof($idcompanies) > 0 ){
                foreach ($idcompanies as $idCompany) {
                    $detail = DetailUserCompany::create([
                        'iduser' => $user->id,
                        'iduser_type' => $params['iduser_type'],
                        'idprivilege' => $params['idprivilege'],
                        'idcompany' => $idCompany,
                    ]);
                    $detail->fresh();
                }
            }

            // $mail = $this->sendMail($params);
            // if(!$mail->original['status']) return $mail;

            DB::commit();
            return $this->successResponse('Usuario creado satisfactoriamente.', $user);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de usuario.', $th->getMessage());
        }
    }

    public function update($id,$params)
    {
        DB::beginTransaction();
        try {
            $validate = $this->validateUser($id, $params);
            if(!$validate->original['status']) return $validate;

            $user = User::find($id);
            $user->rut = $params['rut'];
            $user->name = $params['name'];
            $user->first_surname = $params['first_surname'];
            $user->second_surname = $params['second_surname'];
            $user->gender = $params['gender'];
            $user->type_person = $params['type_person'];
            $user->age = $params['age'];
            $user->birthdate = $params['birthdate'];
            $user->email = $params['email'];
            $user->phone = $params['phone'];
            $user->save();

            DB::commit();
            return $this->successResponse('Usuario actualizado satisfactoriamente.', $user);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un mecanico.', $th->getMessage());
        }
    }

    public function changeStatus($id, $params)
    {
        DB::beginTransaction();
        try {
            $validator = Validator::make($params->all(),[
                'idcompany'    => 'required',
                'status'    => 'required',
             ]);
             if($validator->fails()) return $this->successResponse($validator->errors());

            # Verificar validez del usuario
            $validate = $this->validateUser($id);
            if (!$validate->original['status']) return $validate;

            if($params['status'] == '2') return $this->successResponse('Solo se puede habilitar o deshabilitar.');

            $detail = DetailUserCompany::where('iduser', $id)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if(!$detail) return $this->successResponse('No se encontró al usuario.');

            $detail->status = $params['status'];
            $detail->save();

            DB::commit();
            return $this->successResponse('Estado actualizado satisfactoriamente.', $detail);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un usuario.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            $validate = $this->validateUser($id);
            if(!$validate->original['status']) return $validate;

            $user = User::find($id);
            $user->status = '2';
            $user->save();

            DB::commit();
            return $this->successResponse('Usuario Eliminado satisfactoriamente.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un mecanico.', $th->getMessage());
        }
    }

    public function changePassword($params)
    {
        DB::beginTransaction();
        try {
            $password = JWTAuth::user()->encrypted_password;
            $pass = Crypt::decryptString($password);

            if ($pass == $params['password']) {
                $new_pass = $params['new_password'];

                if ($new_pass != $params['password']) {
                    $user = User::updateOrCreate(['id' => JWTAuth::user()->id],[
                        'password' => bcrypt($params['new_password']),
                        'encrypted_password' => Crypt::encryptString($params['new_password']),
                        'status_confirm' => 1
                    ]);
                    $user->fresh();
                    DB::commit();
                    return $this->successResponse('Contraseña actualizada exitosamente');
                }
                return $this->errorResponse("La contraseña anterior y la contraseña nueva son las mismas", 400);
            }
            return $this->errorResponse("La contraseña no es correcta", 400);
        } catch (\Throwable $th) {
            DB::rollback();
            return $this->externalError('durante el cambio de contraseña.', $th->getMessage());
        }
    }

    // public function forgotPassword($params)
    // {
    //     try {
    //         $user = User::where('rut', $params['rut'])->where('iduser_type', $params['iduser_type'])->first();
    //         if(!$user) return $this->errorResponse("El rut que mandaste no pertenece a ningun usuario con este tipo de usuario", 400);

    //         $email      = $user->email;
    //         $rut        = $user->rut;
    //         $password   = Crypt::decryptString($user->encrypted_password);
    //         $typeUser   = $user->iduser_type;
    //         $url_login  = "https://www.tvp.cl/login?rut=$rut&pass=$password";

    //         switch ($params['iduser_type']) {
    //             case '1':
    //                 $typeUser = 'Super Administrador';
    //                 $user = 'Super Administrador';
    //                 break;
    //             case '2':
    //                 $typeUser = 'Administrador';
    //                 $user = AdminCompany::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '3':
    //                 $typeUser = 'Supervisor';
    //                 $user = Supervisors::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '4':
    //                 $typeUser = 'Jefe de Mantenimiento';
    //                 $user = BossMaintenance::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '5':
    //                 $typeUser = 'Conductor';
    //                 $user = Drivers::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '6':
    //                 $typeUser = 'Mecanico';
    //                 $user = Mechanics::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;

    //             case '7':
    //                 //     $typeUser = 'Invitado';
    //                 //     $user = InvitedsOwner::where('rut', $params['rut'])->first();
    //                 //     $user = $user->name ?? '';
    //                 $user = 'Ayudante';
    //                 break;

    //             default:
    //                 $typeUser = '';
    //                 break;
    //         }

    //         Mail::to($email)->send(new ResendCredentialsMail($user, $rut, $typeUser, $password, $url_login));

    //         return $this->successResponse('Credenciales enviadas a tu correo');
    //     } catch (\Throwable $th) {
    //         return $this->externalError('durante el envió de correo.', $th->getMessage());
    //     }
    // }

    public function validateDataUser($params)
    {
        try {
            $rut = User::rut($params['rut'], $params['iduser_type'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($rut) return $this->errorResponse('El rut ya se encuentra registrado.', 400);

            $email = User::email($params['email'], $params['iduser_type'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($email) return $this->errorResponse('El correo ya se encuentra registrado.', 400);

            $userType = UserType::activeForID($params['iduser_type'])->where('idcompany', $params['idcompany'])->first();
            if (!$userType) return $this->errorResponse('Tipo de usuario no encontrado.', 400);

            return $this->successResponse('OK');
        } catch (\Throwable $th) {
            return $this->externalError('durante la creación de usuario.', $th->getMessage());
        }
    }

    public function validateUser($id, $params = null)
    {
        try {
            $user = User::where('id', $id)->where('status', 1)->first();
            if (!$user) return $this->errorResponse('El usuario no existe.', 400);

            // if(isset($params['iduser_type'])){
            //     $userType = UserType::activeForID($params['iduser_type'])->first();
            //     if (!$userType) return $this->errorResponse('Tipo de usuario no encontrado.', 400);

            //     $rut = User::rut($params['rut'], $params['iduser_type'], $id)->where('status', 1)->first();
            //     if ($rut) return $this->errorResponse('El rut ya se encuentra registrado.', 400);

            //     $email = User::email($params['email'], $params['iduser_type'], $id)->where('status', 1)->first();
            //     if ($email) return $this->errorResponse('El correo ya se encuentra registrado.', 400);
            // }

            return $this->successResponse('OK');
        } catch (\Throwable $th) {
            return $this->externalError('durante la actualización de un mecanico.', $th->getMessage());
        }
    }

    // public function sendMail($params)
    // {
    //     try {
    //         switch ($params['iduser_type']) {
    //             case '1':
    //                 $typeUser = 'Super Administrador';
    //                 $user = 'Super Administrador';
    //                 break;
    //             case '2':
    //                 $typeUser = 'Administrador';
    //                 $user = AdminCompany::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '3':
    //                 $typeUser = 'Supervisor';
    //                 $user = Supervisors::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '4':
    //                 $typeUser = 'Jefe de Mantenimiento';
    //                 $user = BossMaintenance::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '5':
    //                 $typeUser = 'Conductor';
    //                 $user = Drivers::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;
    //             case '6':
    //                 $typeUser = 'Mecanico';
    //                 $user = Mechanics::rut($params['rut'])->where('status', 1)->first();
    //                 $user = $user ? "$user->name . $user->surname" : $typeUser;
    //                 break;

    //             case '7':
    //                 //     $typeUser = 'Invitado';
    //                 //     $user = InvitedsOwner::where('rut', $params['rut'])->first();
    //                 //     $user = $user->name ?? '';
    //                 $user = 'Ayudante';
    //                 break;

    //             default:
    //                 $typeUser = '';
    //                 break;
    //         }

    //         $rut        = $params['rut'];
    //         $password   = $params['password'];
    //         $url_login  = "https://www.tvp.cl/login?rut=$rut&pass=$password";

    //         Mail::to($params['email'])->send(new WelcomeUserMail($user, $rut, $typeUser, $password, $url_login));

    //         return $this->successResponse('Correo electrónico enviado satisfactoriamente.');
    //     } catch (\Throwable $th) {
    //         // return $this->errorResponse('Error al enviar correo.', 400, $th->getMessage());
    //         return $this->externalError('durante el envió de correo.', $th->getMessage());
    //     }
    // }

    public function registerDetailUser($params) {
        try {
            $detailUser = DetailUserCompany::create([
                'iduser'        => $params['iduser'],
                'email'         => $params['email'],
                'phone'         => $params['phone'] ?? null,
                'iduser_type'   => $params['iduser_type'],
                'idprivilege'   => $params['idprivilege'],
                'idcompany'     => $params['idcompany']
            ]);

            return $detailUser;
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function registerUser($rut, $email, $iduser_type, $idcompany, $username = null)
    {
        DB::beginTransaction();
        try {
            ## Crear un usuario para el vendedor

            # Primero verifico si es que ya hay un usuario con el mismo rut
            $verifyUser = User::rut($rut)->active()->first();

            # En caso encontrar solo adiciono el idcompany a su detalle
            if ($verifyUser) {
                # Busco si tiene un detalle creado con el mismo idcompany
                $userCompany = DetailUserCompany::findUser($verifyUser->id, $iduser_type)->where('idcompany', $verifyUser->idcompany)->active()->first();
                if(!$userCompany) {
                    DetailUserCompany::create([
                        'iduser'            => $verifyUser->id,
                        'iduser_type'       => $iduser_type,
                        'idprivilege'       => 1, #Por revisar más adelante
                        'idcompany'         => $verifyUser->idcompany
                    ]);
                }

                # Mandar otro correo comentando que ya tiene un usuario, solo se habilitó para otro rol

            } else {
                # En caso no encontrar creo desde cero
                // $password = Str::random(10);
                $password = 'demo';
                $user = User::create([
                    'rut'                   => $rut,
                    'email'                 => $email,
                    'username'              => $username,
                    'password'              => bcrypt($password),
                    'encrypted_password'    => Crypt::encryptString($password),
                    'idcompany'             => $idcompany
                ]);

                DetailUserCompany::create([
                    'iduser'            => $user->id,
                    'iduser_type'       => 3,
                    'idprivilege'       => 1,
                    'idcompany'         => $user->idcompany
                ]);

                # Enviar un correo de bienvenida
            }
            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack();
        }
    }
}
