<?php

namespace App\Services;

use App\Http\Resources\TypeProductResource;
use App\Models\TypeProduct;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class TypeProductService
 * @package App\Services
 */
class TypeProductService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $types = TypeProduct::typeProductFilters()->orderBy('id', 'desc');

        $types = !empty($withPagination)
            ? $types->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $types->get();

        $types = TypeProductResource::collection($types);

        return $this->successResponse('Lectura exitosa.', $types);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            #Validar duplicidad de código
            $code = TypeProduct::where('code', $params['code'])->active()->first();
            if ($code) return $this->errorResponse('El código ingresado ya se encuentra registrado.', 400);

            #Validar duplicidad de nombre
            $name = TypeProduct::where('name', $params['name'])->active()->first();
            if ($name) return $this->errorResponse('El nombre ingresado ya se encuentra registrado.', 400);

            $type = TypeProduct::create([
                'code' => $params['code'],
                'name' => $params['name'],
                'description' => $params['description'] ?? null
            ]);
            $type->save();
    
            DB::commit();
            return $this->successResponse('Tipo de producto registrado satisfactoriamente.', $type);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un tipo de producto.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            #Validar tipo de produto
            $type = TypeProduct::activeForID($id)->first();
            if (!$type) return $this->errorResponse('Tipo de producto no válido.', 400);
            
            #Validad duplicidad de código
            $code = TypeProduct::where('id', '!=', $id)->where('code', $params['code'])->active()->first();
            if ($code) return $this->errorResponse('El código ingresado ya se encuentra registrado.', 400);

            #Validar duplicidad de nombre
            $name = TypeProduct::where('id', '!=', $id)->where('name', $params['name'])->active()->first();
            if ($name) return $this->errorResponse('El nombre ingresado ya se encuentra registrado.', 400);

            $type->update($params);
            $type->save();

            DB::commit();
            return $this->successResponse('Tipo de producto actualizado satisfactoriamente.', $type);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un tipo de producto.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            #Validar tipo de produto
            $type = TypeProduct::activeForID($id)->first();
            if (!$type) return $this->errorResponse('Tipo de producto no válido.', 400);

            $type->status = 2;
            $type->save();

            DB::commit();
            return $this->successResponse('Tipo de producto eliminado satisfactoriamente.', $type);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un tipo de producto.', $th->getMessage());
        }
    }
}
