<?php

namespace App\Services;

use App\Models\TypeAttribute;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class TypeAttributeService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $typeAttributes = TypeAttribute::where('status', '!=', 2)->orderBy('id', 'desc');

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $typeAttributes->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $typeAttributes = !empty($withPagination)
            ? $typeAttributes->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $typeAttributes->get();

        return $this->successResponse('Lectura exitosa.', $typeAttributes);
    }

    public function listByCompany($params, $withPagination)
    {
        $typeAttributes = TypeAttribute::where('status', '!=', 2)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $typeAttributes->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $typeAttributes->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $typeAttributes = !empty($withPagination)
            ? $typeAttributes->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $typeAttributes->get();

        return $this->successResponse('Lectura exitosa.', $typeAttributes);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            
            $nameExisting = TypeAttribute::where('name', $params['name'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $codeExisting = TypeAttribute::where('code', $params['code'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($codeExisting) return $this->errorResponse('El código ya se encuentra registrado.', 400);

            $usersession = Auth::user();

            $productAttribute = TypeAttribute::create([
                'code' => $params['code'],
                'name' => $params['name'],
                'description' => $params['description'],
                'createdby' => $usersession->id,
                'status' => isset($params['status']) ? $params['status'] : 1,
                'idcompany' => $params['idcompany'],
            ]);
            $productAttribute->save();

            DB::commit();
            return $this->successResponse('Tipo atributo de producto creado satisfactoriamente.', $productAttribute);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un Atributo.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la bodega
            $validate = $this->verifyAttribute($id);
            if (!$validate->original['status']) return $validate;

            $nameExisting = TypeAttribute::where('name', $params['name'])->where('id','!=', $id)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $codeExisting = TypeAttribute::where('code', $params['code'])->where('id','!=', $id)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($codeExisting) return $this->errorResponse('El código ya se encuentra registrado.', 400);

            $productAttribute = TypeAttribute::find($id);
            $productAttribute->code = $params['code'];
            $productAttribute->name = $params['name'];
            $productAttribute->description = $params['description'];
            $productAttribute->save();

            DB::commit();
            return $this->successResponse('Tipo de tributo de producto actualizado satisfactoriamente.', $productAttribute);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un Atributo.', $th->getMessage());
        }
    }

    public function changeStatus($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la bodega
            $validate = $this->verifyAttribute($id);
            if (!$validate->original['status']) return $validate;

            if($params['status'] == '2') return $this->successResponse('Solo se puede habilitar o deshabilitar.');

            $productAttribute = TypeAttribute::find($id);
            $productAttribute->status = $params['status'];
            $productAttribute->save();

            DB::commit();
            return $this->successResponse('Estado actualizado satisfactoriamente.', $productAttribute);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un Atributo.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la bodega
            $validate = $this->verifyAttribute($id);
            if (!$validate->original['status']) return $validate;

            $productAttribute = TypeAttribute::find($id);
            $productAttribute->status = 2;
            $productAttribute->save();

            DB::commit();
            return $this->successResponse('Tipo de atributo de producto eliminado satisfactoriamente.', $productAttribute);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un Atributo.', $th->getMessage());
        }
    }

    private function verifyAttribute($idattribute)
    {
        $productAttribute = TypeAttribute::where('id', $idattribute)->first();
        if (!$productAttribute) return $this->errorResponse('No se encontró el tipo de atributo.', 400);

        return $this->successResponse('OK');
    }
}
