<?php

namespace App\Services;

use App\Http\Resources\PayResource;
use App\Mail\MembershipPay;
use App\Mail\WelcomeCompanyFirstPayMail;
use App\Models\Company;
use App\Models\CompanyDefaultPermissions;
use App\Models\DetailMembershipPayment;
use App\Models\DetailUserCompany;
use App\Models\MembershipCosts;
use App\Models\Pay;
use App\Models\PermissionsUser;
use App\Models\Ticket;
use App\Models\Url;
use App\Models\User;
use App\Traits\HasResponse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use Transbank\Webpay\WebpayPlus\Transaction;
use Transbank\Webpay\Options;

class TransbankService
{
    use HasResponse;
    /** @var DetailMembershipPaymentService */
    private $detailMembershipPaymentService;

    public function __construct(DetailMembershipPaymentService $detailMembershipPaymentService)
    {
        $this->detailMembershipPaymentService = $detailMembershipPaymentService;
    }

    private function Urls($id)
    {
        $urls = Url::find($id)->url;
        return $urls;
    }

    public function startWebpayPlusTransaction($id, $params)
    {
        try {
            # Validar el tipo de pago que se hará
            switch ($params['type']) {
                case '1':
                    # Ticket, verificar que sea válido
                    $ticket = Ticket::activeForID($id)->where('status_ticket', 1)->first();
                    if (!$ticket) return $this->errorResponse('El ticket seleccionado es inválido.', 400);
                    if ($ticket->payment_status == 1) return $this->errorResponse('El ticket seleccionado ya fue pagado.', 400);
                    break;

                case '2':
                    # Pago de membresías
                    $membershipCosts = MembershipCosts::activeForID($id)->first();
                    if (!$membershipCosts) return $this->errorResponse('La membresía seleccionada es inválida.', 400);
                    break;
            }

            # Cargar pago
            $url = $this->payWebPay($id, $params['type']);
            if (!$url->original['status']) return $this->errorResponse($url->original['data']['message'], $url->original['code']);

            return $this->successResponse('Generación de pago exitoso.', $url->original['data']['detail']);
        } catch (\Throwable $th) {
            throw $th;
            return $this->externalError('durante el pago.', $th->getMessage());
        }
    }

    public function payWebPay($id, $type)
    {
        try {
            DB::beginTransaction();
            $iduser = Auth::user()->id;
            # Validar el tipo de pago que se realiza
            switch ($type) {
                case '1':
                    # Pago de ticket
                    $register               = Ticket::find($id);
                    $purchase_identifier    = 'T-' . $register->id;
                    $amount                 = $register->net_total;
                    $idcompany              = $register->idcompany;
                    $payType                = 1;
                    break;

                case '2':
                    # Pago de membresías
                    $register               = MembershipCosts::find($id);
                    $purchase_identifier    = 'M-' . $register->id;
                    $amount                 = $register->amount;
                    $idcompany              = request('idcompany');
                    $payType                = 2;
                    break;
            }

            # Información de pago de acuerdo a la compañía
            $company = Company::activeForID($idcompany)->first();
            if (!$company) return $this->errorResponse('La compañía es inválida.', 400);

            # Validar a quién se realizará el pago
            switch ($type) {
                case '1':
                    # Claves para el ticket
                    if (is_null($company->code_comerce_wp) || is_null($company->api_key_wp)) return $this->errorResponse('La compañía no cuenta con credenciales de webpay.', 400);
                    $code_comerce_wp = $company->code_comerce_wp;
                    $api_key_wp = $company->api_key_wp;
                    break;

                case '2':
                    # Claves para el pago de membresías
                    $code_comerce_wp = $company->code_comerce_wp;
                    $api_key_wp = $company->api_key_wp;
                    break;
            }

            # Entorno de producción y local
            if (getenv('APP_ENV') == 'production') {
                $options = Options::forProduction($code_comerce_wp, $api_key_wp);
            } else {
                $options = null;
            }

            # Inicio de la transacción para obtener el URL
            $transaccion = (new Transaction($options))->create(
                $purchase_identifier,       # buy_order
                $iduser,                    # session_id
                $amount,                    # amount
                route('confirmar_pago_wp')  # return_url
            );

            $url = $transaccion->getUrl() . '?token_ws=' . $transaccion->getToken();

            # Generar un registro en la tabla pagos
            Pay::create([
                'pay_type'              => $payType,
                'idregister'            => $register->id,
                'iduser'                => $iduser,
                'amount'                => $amount,
                'purchase_identifier'   => $purchase_identifier,
                'token'                 => $transaccion->getToken()
            ]);

            DB::commit();
            return $this->successResponse('Carga de datos en Web Pay Plus Mall con éxito.', $url);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la carga del pago.', $th->getMessage());
        }
    }

    public function confirmar_pago_wp($params)
    {
        try {
            # En caso de respuesta Aceptada o Rechazada
            if (isset($params['token_ws']) && !isset($params['TBK_TOKEN'])) {

                # Lectura del registro a actualizar por ser una operación exitosa
                $pay            = Pay::where('token', $params['token_ws'])->first();
                $confirmation   = (new Transaction)->commit($params['token_ws']);
                $buy_order      = $confirmation->getBuyOrder();
                $iduser         = $confirmation->getSessionId();
                $idcompany      = User::find($iduser)->idcompany;

                # Verificar el tipo de transacción que se realiza
                $parts = explode('-', $buy_order);

                if (count($parts) === 2) {
                    $type       = $parts[0];
                    $idregister = $parts[1];
                } else {
                    return $this->errorResponse('Ocurrió un error en la verificación del pago.', 400);
                }

                # Verificar según el tipo
                switch ($type) {
                    case 'T':
                        $register   = Ticket::find($idregister);
                        $company    = Company::find($register->idcompany);

                        # Url
                        $paySucces = $this->Urls(2) . $this->Urls(3) . $company->id;
                        break;
                    case 'M':
                        $register   = MembershipCosts::find($idregister);
                        $company    = Company::find($idcompany);
                        # Url
                        $paySucces = $this->Urls(2) . $this->Urls(6) . $company->id;
                        break;
                }

                if ($confirmation && $confirmation->isApproved()) {

                    # Actualizar el estado de pago
                    $pay->update([
                        'transaction_status' => 1,
                        'status' => 1
                    ]);
                    $pay = $pay->fresh();

                    switch ($type) {
                        case 'T':
                            # Para actualizar registros del ticket y pago por ser una operación exitosa
                            $register->update(['payment_status' => 1]);
                            $register = $register->fresh();
                            break;

                        case 'M':
                            # Acciones en pago de membresía
                            $create['idcompany']         = $idcompany;
                            $create['idmembershipcosts'] = $idregister;
                            $create['idpay']             = $pay->id;

                            $user   =  User::where('idcompany', $idcompany)->active()
                                ->whereHas('detailUser', function ($q) use ($idcompany) {
                                    $q->where('iduser_type', 2)->where('idcompany', $idcompany)->active();
                                })->first();

                            $adminName = $user->username;

                            # Mandar correo de bienvenido al sistema por hacer su primer pago, informando que ya se encuentra operativo
                            $detailMembershipPayment = DetailMembershipPayment::where('idcompany', $idcompany)->first();
                            if (!$detailMembershipPayment) {

                                $rut_company    = base64_encode($company->rut);
                                $username       = base64_encode($user->username);
                                $rut            = base64_encode($user->rut);
                                $password       = base64_encode(Crypt::decryptString($user->encrypted_password));
                                $iduser_type    = base64_encode(2);
                                $url_autologin  = Url::find(2)->url . "/login?rut_company=$rut_company&username=$username&rut=$rut&password=$password&iduser_type=$iduser_type";

                                # CORREO DE BIENVENIDA
                                Mail::to($company->email)->send(new WelcomeCompanyFirstPayMail($company->business_name, $url_autologin));
                            }

                            # Verificar si el pago que se esta realizando es bajo una membresía diferente al actual de la empresa
                            $membershipCosts = $register;
                            if ($membershipCosts->idmembership != $company->idmembership) {

                                # Data de los permisos
                                switch ($membershipCosts->idmembership) {
                                    case '1':
                                        $data = config("plans.plans.basic.permissions", []);
                                        break;
                                    case '2':
                                        $data = config("plans.plans.standar.permissions", []);
                                        break;
                                    case '3':
                                        $data = config("plans.plans.pro.permissions", []);
                                        break;
                                }
                                $admin      = $data['admin'];
                                $seller     = $data['seller'];
                                $supplier   = $data['supplier'];
                                $customer   = $data['customer'];

                                $dataPermissions = [
                                    2 => $admin,
                                    3 => $seller,
                                    4 => $supplier,
                                    5 => $customer,
                                ];

                                # Actualizo todos los permisos por defecto de la empresa
                                $companyPermissionsDefaults = CompanyDefaultPermissions::where('idcompany', $company->id)->active()->get();

                                foreach ($companyPermissionsDefaults as $companyPermissionsDefault) {
                                    $typeUser = $companyPermissionsDefault->iduser_type;

                                    if (isset($dataPermissions[$typeUser])) {
                                        $permissionsToUpdate = $dataPermissions[$typeUser];
                                        $companyPermissionsDefault->update(['permissions' => $permissionsToUpdate]);
                                    }
                                }

                                # Reviso si dejaré los permisos tal como estan o quitaré los que no cuenta el nuevo plan
                                if ($membershipCosts->idmembership < $company->idmembership) {

                                    # Coloco los permisos personalizados limitados a los que no deben tener
                                    $customPermissions = PermissionsUser::where('idcompany', $company->id)
                                        ->active()->get();

                                    $detailUsers = DetailUserCompany::where('idcompany', $company->id)
                                        ->whereIn('iduser', $customPermissions->pluck('iduser'))
                                        ->active()
                                        ->get();

                                    foreach ($detailUsers as $detailUser) {

                                        $dataPermissions = [
                                            2 => $admin,
                                            3 => $seller,
                                            4 => $supplier,
                                            5 => $customer,
                                        ][$detailUser->iduser_type] ?? [];

                                        # Obtener la diferencia entro los permisos que tiene con respecto a los nuevos que puede tener (Los nuevos son menos de los que puede tener actualmente)
                                        # Luego quitar los permisos que no puede tener y actualizar sus permisos

                                        $permissionesUser = PermissionsUser::where('iduser', $detailUser->iduser)
                                            ->where('idcompany', $company->id)
                                            ->active()->first();

                                        $differenceBasic = array_values(array_diff($permissionesUser->permissions, $dataPermissions)); # Permisos a quitar
                                        $differenceFinal = array_values(array_diff($permissionesUser->permissions, $differenceBasic)); # Permisos restantes

                                        $permissionesUser->update(['permissions' => $differenceFinal]);
                                    }
                                }
                            }

                            # Generar registro
                            $this->detailMembershipPaymentService->createMembershipPayment($create);

                            # Mandar correo de pago de membresía adjuntando la factura generada
                            $detailMembershipPayment = DetailMembershipPayment::where('idcompany', $idcompany)->latest('id')->get()->load('membershipCosts.plans', 'pay');
                            $detailMembershipPayment = $detailMembershipPayment[0];


                            # Entorno de producción y local
                            if (getenv('APP_ENV') == 'production') {
                                $document = $detailMembershipPayment->url_invoice;
                            } else {
                                # En caso estar modo desarrollo no se genera un pdf, entonces usar por defecto algun otro que este disponible
                                # No olvidar usar el php artisan storage:link
                                $document = 'storage/documentos/doc13_33333333-3.pdf';
                            }

                            Mail::to($user->email)->send(new MembershipPay(
                                $user->rut,
                                $adminName,
                                Carbon::parse($detailMembershipPayment->payment_date)->toDateString(),
                                Carbon::parse($detailMembershipPayment->payment_date_end)->toDateString(),
                                $detailMembershipPayment['membershipCosts']['plans']->month,
                                $detailMembershipPayment['pay']->amount,
                                $document
                            ));

                            $paySucces = $paySucces . '?r=' . base64_encode($user->rut) . '&pw=' . Crypt::decryptString($user->encrypted_password) . '&tp=2';
                            break;
                    }

                    return redirect()->away($paySucces);
                } else {

                    # Actualización de estado de pago
                    $pay->update(['transaction_status' => 2, 'status' => 1]);
                    return redirect()->away($this->Urls(2) . $this->Urls(4) . $company->id);
                }
            }
            # En caso de anulación o errores
            else {
                return redirect()->away($this->Urls(2));
            }
        } catch (\Throwable $th) {
            throw $th;
            return redirect()->away('pagina_de_error_desconocido.com');
        }
    }

    public function assignCredentials($params)
    {
        DB::beginTransaction();
        try {

            # Solo admin
            if (Auth::user()->detailUser->where('iduser_type', '<=', 2)->isEmpty()) return $this->errorResponse('Usuario no autorizado.', 401);

            $company = Company::activeForID($params['idcompany'])->first();
            if (!$company) return $this->errorResponse('Compañía inválida.', 401);

            $company->update($params);

            DB::commit();
            return $this->successResponse('Credenciales de webpay editadas con éxito.', $company);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la edición de las credenciales.', $th->getMessage());
        }
    }

    public function payments($params, $withPagination)
    {
        $payment = Pay::payFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc');

        if (isset($params['start_date']) && isset($params['end_date'])) {
            if ($params['end_date'] < $params['start_date']) return $this->errorResponse('La fecha fin no puede ser menor a la fecha inicial.', 400);

            $payment = $payment->whereBetween('creation_date', [$params['start_date'], $params['end_date']]);
        }

        $payment = !empty($withPagination)
            ? $payment->orderBy('id', 'desc')->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $payment->orderBy('id', 'desc')->get();

        return $this->successResponse('Lectura exitosa.', PayResource::collection($payment->load('user')));
    }
}
