<?php

namespace App\Services;

use App\Http\Resources\SalesBookResource;
use App\Http\Resources\TicketResource;
use App\Models\Company;
use App\Models\DetailShoppingCart;
use App\Models\DetailTicket;
use App\Models\Product;
use App\Models\ShoppingCart;
use App\Models\Ticket;
use App\Models\Supplier;
use App\Traits\HasResponse;
use Carbon\Carbon;
use Illuminate\Http\Exceptions\HttpResponseException;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;

class TicketService
{
    use HasResponse;

    /** @var SoftnetService */
    private $softnetService;

    public function __construct(SoftnetService $softnetService)
    {
        $this->softnetService = $softnetService;
    }

    public function list($withPagination)
    {
        $tickets = Ticket::ticketFilters()->orderBy('id', 'desc');

        $tickets = !empty($withPagination)
            ? $tickets->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $tickets->get();

        $tickets = TicketResource::collection($tickets->load('user', 'customer.district', 'customer.province', 'customer.country', 'company', 'detail.product.product', 'detail.productOrigin', 'pay', 'typeDocument', 'saleChannel', 'wayToPay'));

        return $this->successResponse('Lectura exitosa.', $tickets);
    }

    public function salesBook($withPagination)
    {
        $tickets = Ticket::whereNotNull('folio')
            // ->whereIn('idtype_document', [1, 6])
            ->ticketFilters()->orderBy('id', 'desc');

        $tickets = !empty($withPagination)
            ? $tickets->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $tickets->get();

        $tickets = SalesBookResource::collection($tickets->load('user', 'customer', 'company', 'typeDocument'));

        return $this->successResponse('Lectura exitosa.', $tickets);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            #Validar: Duplicidad de ticket
            // $ticket = Ticket::where('idshopping_cart', $params['idshopping_cart'])->active()->first();
            // if ($ticket) return $this->errorResponse('Ya se ha creado un ticket para el carrito asignado.', 400);

            #Validar que el carrito de compras tenga un cliente seleccionado
            $shoppingCart = ShoppingCart::activeForID($params['idshopping_cart'])->company()->first();
            if (!$shoppingCart) return $this->errorResponse('El carrito de compras no es válido.', 400);

            #Validar que el carrito de compras tenga productos añadidos
            $detShoppingCart = DetailShoppingCart::where('idshopping_cart', $shoppingCart->id)->active()->get();
            if (count($detShoppingCart) == 0) return $this->errorResponse('El carrito de compras no tiene productos añadidos.', 400);

            $customer = Supplier::where('rut', $params['customer_rut'])->active()->first();
            if (!$customer) return $this->errorResponse('El cliente no es valido.', 400);

            #Generar numero de ticket
            $nro_ticket = Ticket::orderBy('id', 'desc')->get();
            $nro_ticket = count($nro_ticket) > 0 ? count($nro_ticket) + 1 : 1;

            $discountGlobalMount = $shoppingCart->discounttype === 2 ? ($shoppingCart->sub_total * ($shoppingCart->global_discount/100)) :  $shoppingCart->global_discount;

            $ticket = Ticket::create([
                'date'              => Carbon::now()->toDateString(),
                'nro_ticket'        => $nro_ticket,
                'iduser_created'    => $shoppingCart->iduser_created,
                'idcustomer'        => $customer->id,
                'idclient_address'  => $params['idclient_address'] ?? null,
                'seller_rut'        => $params['seller_rut'] ?? null,
                'gross_total'       => $shoppingCart->gross_total,
                'net_total'         => $shoppingCart->net_total,
                'IVA'               => $shoppingCart->IGV,
                'idcompany'         => $shoppingCart->idcompany,
                'idtype_document'   => $params['idtype_document'],
                'idsale_channel'    => $params['idsale_channel'] ??  null,
                // 'payment_condition' => $params['payment_condition'],
                'idway_to_pay'      => $params['idway_to_pay'],
                'bussines_area'     => $params['bussines_area'],
                'observation'       => $params['observation'],
                'folio'             => $params['folio'] ?? null,
                'total_discount'    => $discountGlobalMount,
                'sub_total'         => $shoppingCart->sub_total,
                'total'             => $shoppingCart->total
            ]);

            #Crear Detalle
            $details = [];
            foreach ($detShoppingCart as $detail) {
                $details[] = [
                    'idticket' => $ticket->id,
                    'idpricelist_product' => $detail->idpricelist_product,
                    'idproduct' => $detail->idproduct,
                    'amount' => $detail->quantity,
                    'price' => $detail->price,
                    'discount' => $detail->discount,
                    'discounttype' => $detail->discounttype,
                    'gross_subtotal' => $detail->gross_subtotal,
                    'net_subtotal' => $detail->net_subtotal,
                ];
            }

            if (!empty($details)) DetailTicket::insert($details);

            $shoppingCart->update(['status_ticket' => 1]);

            DB::commit();
            return $this->successResponse('Ticket creado con éxito.', $ticket);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un ticket.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            #Validar carrito de compras
            $ticket = Ticket::activeForID($id)->company()->first();
            if (!$ticket) return $this->errorResponse('Ticket no válido.', 400);

            $ticket->update(['status' => 2]);

            #Eliminar detalles
            DetailTicket::where('idticket', $ticket->id)->active()->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('Ticket eliminado con éxito.', $ticket);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminado de un ticket.', $th->getMessage());
        }
    }

    public function generateDocument($params)
    {
        try {
            #Validar parámetros
            $validator = Validator::make($params, [
                'idticket'  => 'required|numeric|validate_ids_exist:Ticket',
                'idcompany' => 'required|numeric|validate_ids_exist:Company',
                'folio'     => 'nullable|numeric|min:1',

                'references' => 'nullable|array|min:1',
                'references.*.TpoDocRef' => 'required|numeric',
                'references.*.FolioRef'  => 'required|numeric|min:0',
                'references.*.FchRef'    => 'required|date',
                'references.*.CodRef'    => 'required|numeric|in:1,2,3',
                'references.*.RazonRef'  => 'required|min:5|max:80'
            ]);
            if ($validator->fails()) {
                $errors = $validator->errors();
                $primerArray = reset($errors);
                $primerError = reset($primerArray);
                $message = $primerError[0];

                throw new HttpResponseException($this->errorResponse($message, 400));
            }

            #Validar: Ticket con documento emitido
            $ticket = Ticket::activeForId($params['idticket'])->where('status_invoice', 1)->first();
            if ($ticket) return $this->errorResponse('Ya se ha emitido el comprobante del ticket seleccionado.', 400);

            $company = Company::activeForID($params['idcompany'])->get()->load('district');
            $ticket  = Ticket::activeForId($params['idticket'])->company()->get()->load('customer.district', 'detail');

            $customer           = $ticket[0]['customer'];
            $detailsTicket      = $ticket[0]['detail'];
            $districtCustomer   = $ticket[0]['customer']['district'];
            $ticket             = $ticket[0];
            $disctrictCompany   = $company[0]['district'];
            $company            = $company[0];

            #Obtener el token del usuario erp de la empresa
            // $userErp = UserERP::where('id', $company->rut)->first(); #descomentar
            $userErp = (object)[
                'rut'                => '22222222-2',
                'user'               => 'usuario',
                'password'           => 'demoerp',
                'encrypted_password' => Crypt::encryptString('demoerp'),
                'status'             => 1
            ];
            if (!$userErp) return $this->errorResponse('La empresa no registra su usuario del ERP para comenzar a facturar.', 400);
            $token    = $this->softnetService->getTokenSoftnet($userErp);
            $tokenErp =  Crypt::encryptString($token);

            $param['token_erp'] = $tokenErp;

            $TipoDTE = 0;
            switch ($ticket->idtype_document) {
                    #boleta
                case 1:
                    $TipoDTE = 39;
                    break;

                    #factura
                case 2:
                    $TipoDTE = 33;
                    break;

                    #nota de credito
                case 3:
                    $TipoDTE = 61;
                    break;

                    #nota de debito
                case 4:
                    $TipoDTE = 56;
                    break;

                    #cotizacion
                case 5:
                    $TipoDTE = 777;
                    break;

                default:
                    break;
            }

            $Encabezado = (object)[
                'IdDoc' => (object)[
                    'TipoDTE' => $TipoDTE,
                    'Folio'   => $params['folio'] ?? 1,
                    'FchEmis' => Carbon::now()->toDateString(),
                ],
                'Emisor' => (object)[
                    'RUTEmisor'     => $company->rut,
                    'RznSocEmisor'  => $company->business_name,
                    'GiroEmis'      => $company->giro,
                    'Telefono'      => $company->phone,
                    'Acteco'        => 722000, #Por consultar y modificar
                    'DirOrigen'     => $company->address,
                    'CmnaOrigen'    => $disctrictCompany->name,
                    'CiudadOrigen'  => $disctrictCompany->name
                ],
                'Receptor' => (object)[
                    'RUTRecep'      => $customer->rut,
                    'RznSocRecep'   => $customer->name_rz,
                    'GiroRecep'     => $customer->giro,
                    'DirRecep'      => $customer->address,
                    'CmnaRecep'     => $districtCustomer->name,
                    'CiudadRecep'   => $districtCustomer->name
                ],
                'Totales' => (object)[
                    'MntNeto'   => round($ticket->net_total),
                    'IVA'       => round($ticket->IVA),
                    'MntTotal'  => round($ticket->gross_total)
                ]
            ];

            #Si es factura agregar monto excento a totales
            if ($ticket->idtype_document == 2) {
                $Encabezado->Totales->MntExe = $ticket->mnt_excent;
            }

            $detalles = [];
            foreach ($detailsTicket as $row) {
                $NmbItem = Product::whereHas('detailPriceListProduct', fn ($q) => $q->where('id', $row->idpricelist_product))->value('name');
                $detalles[] = (object)[
                    'NmbItem'   => $NmbItem,
                    'PrcItem'   => strval($row->price),
                    'QtyItem'   => strval($row->amount),
                    'MontoItem' => strval($row->price * $row->amount)
                ];
            }

            $Documento = (object)[
                'Encabezado' => $Encabezado,
                'Detalle'    => $detalles
            ];

            #Si es nota de credito o debito agregar referencias
            if ($ticket->idtype_document == 3 || $ticket->idtype_document == 4) {
                $referencias = [];
                foreach ($params['references'] as $row) {
                    $referencias[] = (object)[
                        'TpoDocRef' => $row->TpoDocRef,
                        'FolioRef'  => $row->FolioRef,
                        'FchRef'    => $row->FchRef,
                        'CodRef'    => $row->CodRef,
                        'RazonRef'  => $row->RazonRef
                    ];
                }
                $Documento->Referencia = $referencias;
            }

            $DTE = [
                'DTE' => [
                    'Documento'     => $Documento,
                    'Adicionales'   => (object)[]
                ]
            ];

            $facturaERP = $this->softnetService->EmiteFactura($param, $DTE, $ticket->id, $ticket->idtype_document);
            if (!isset($facturaERP['mensaje']) || $facturaERP['mensaje'] != "Dte generado con exito.") return $this->errorResponse("Error al crear el documento.", 400);

            $urlpdfDTE = $this->softnetService->consultaDTE($param, $facturaERP['tipo'], $facturaERP['folio'], $company->rut);

            $ticket = Ticket::find($params['idticket'])->first();

            if (!empty($urlpdfDTE)) {
                $pdfContent = file_get_contents($urlpdfDTE);

                $folder = "$company->business_name/documentos/";
                $filename = 'doc' . $facturaERP['folio'] . '_' . $customer->rut . '.pdf';
                $filePath = $folder . '/' . $filename;

                if (!Storage::disk('public')->exists($folder)) Storage::disk('public')->makeDirectory($folder);
                Storage::disk('public')->put($filePath, $pdfContent);

                $ticket->update([
                    'status_invoice' => 1,
                    'url_invoice'    => "storage/$filePath",
                    'folio'          =>  $facturaERP['folio']
                ]);
            } else {
                $ticket->update([
                    'status_invoice' => 1,
                    'folio'          =>  $facturaERP['folio']
                ]);
            }

            return $this->successResponse('Documento emitido con éxito');
        } catch (\Throwable $th) {
            throw $th;
        }
    }
}
