<?php

namespace App\Services;

use App\Http\Resources\SupplierResource;
use App\Models\Company;
use App\Models\Country;
use App\Models\Department;
use App\Models\District;
use App\Models\PriceList;
use App\Models\Province;
use App\Models\Salesman;
use App\Models\Segment;
use App\Models\Supplier;
use App\Models\SupplierContact;
use App\Models\ClientAddress;
use App\Models\WayToPay;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class SupplierService
 * @package App\Services
 */
class SupplierService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $suppliers = Supplier::supplierFilters()->orderBy('id', 'desc');

        $suppliers = !empty($withPagination)
            ? $suppliers->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $suppliers->get();

        $suppliers = SupplierResource::collection($suppliers->load('wayToPay', 'segment', 'country', 'department', 'province', 'district', 'company'));

        return $this->successResponse('Lectura exitosa.', $suppliers);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            #Validar duplicidad de rut
            $rut = Supplier::where('rut', $params['rut'])->active()->first();
            if ($rut) return $this->errorResponse('Ya existe un registro con el rut ingresado.', 400);
            
            $supplier = Supplier::create([
                'rut'           => $params['rut'],
                'name_rz'       => $params['name_rz'],
                'fantasy_name'  => $params['fantasy_name'] ?? null,
                'web'           => $params['web'] ?? null,
                'giro'          => $params['giro'] ?? null,
                'email'         => $params['email'],
                'phone'         => $params['phone'],
                'fax'           => $params['fax'] ?? null,
                'type'          => $params['type'],
                'legal_nature'  => $params['legal_nature'],
                'idcompany'     => $params['idcompany'],
            ]);
            $supplier->save();

            if (count($params['contacts']) > 0) {
                $contacts = [];
                foreach ($params['contacts'] as $contact) {
                    $contacts[] = [
                        'rut' => $contact['rut'],
                        'name' => $contact['name'],
                        'position' => $contact['charge'],
                        'email' => $contact['email'] ?? null,
                        'phone' => $contact['phone'] ?? null,
                        'idsupplier' => $supplier->id,
                        'idcompany' => $supplier->idcompany,
                    ];
                }
                SupplierContact::insert($contacts);
            }

            if(count($params['addresses']) > 0) {

                if (count($params['addresses']) > 0) {
                    $addresses = array_map(function ($address) use ($supplier) {
                        $address['idclient'] = $supplier->id;
                        return $address;
                    }, $params['addresses']);
        
                    ClientAddress::insert($addresses);
                }

            }

            DB::commit();
            return $this->successResponse('Proveedor/cliente registrado satisfactoriamente.', $supplier);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un proveedor/cliente.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            #Validar proveedor/cliente
            $supplier = Supplier::activeForId($id)->first();
            if (!$supplier) return $this->errorResponse('Proveedor/cliente no válido.', 400);

            #Validar duplicidad de rut
            $rut = Supplier::where('id', '!=', $id)->where('rut', $params['rut'])->active()->first();
            if ($rut) return $this->errorResponse('Ya existe un registro con el rut ingresado.', 400);
            
            $supplier->update($params);
            $supplier->save();

            $idContact = [];
            $newContacts = [];
            foreach ($params['contacts'] as $contact) {
                if ($contact['id'] != null) {
                    $idContact[] = (int) $contact['id'];
                } else {
                    $newContacts[] = [
                        'rut' => $contact['rut'],
                        'name' => $contact['name'],
                        'position' => $contact['charge'],
                        'email' => $contact['email'] ?? null,
                        'phone' => $contact['phone'] ?? null,
                        'idsupplier' => $supplier->id,
                        'idcompany' => $supplier->idcompany,
                    ];
                }
            }

            #Borrar contactos
            SupplierContact::whereNotIn('id', $idContact)->where('idsupplier', $supplier->id)->active()->update(['status' => 2]);

            #Guardar contactos
            if (!empty($newContacts)) {
                SupplierContact::insert($newContacts);
            }

            $idAddresses = [];
            $newAddresses = [];

            foreach ($params['addresses'] as $clientAddress) {
                if($clientAddress['id'] != null) {
                    $idAddresses[] = (int) $clientAddress['id'];
                } else {
                    $newAddresses[] = $clientAddress;
                }
            }

            ClientAddress::whereNotIn('id', $idAddresses)->where('idclient', $supplier->id)->active()->update(['status' => 2]);
            if (!empty($newAddresses)) {
                $addresses = array_map(function ($address) use ($supplier) {
                    $address['idclient'] = $supplier->id;
                    return $address;
                }, $newAddresses);
                ClientAddress::insert($addresses);
            }


            DB::commit();
            return $this->successResponse('Proveedor/cliente actualizado satisfactoriamente.', $supplier);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la edición de un proveedor/cliente.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        try {
            #Validar proveedor/cliente
            $supplier = Supplier::activeForId($id)->first();
            if (!$supplier) return $this->errorResponse('Proveedor/cliente no válido.', 400);

            $supplier->status = 2;
            $supplier->save();

            SupplierContact::where('idsupplier', $supplier->id)->update(['status' => 2]);
            SupplierContact::where('idclient', $supplier->id)->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('Proveedor/cliente actualizado satisfactoriamente.', $supplier);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la edición de un proveedor/cliente.', $th->getMessage());
        }
    }
}
