<?php

namespace App\Services;

use App\Http\Resources\SupervisorsResource;
use App\Models\Supervisors;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

class SupervisorService
{
    use HasResponse;

    public function list($withPagination)
    {
        $supervisor = Supervisors::supervisorFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc');

        $supervisor = !empty($withPagination)
            ? $supervisor->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $supervisor->get();

        $supervisor = SupervisorsResource::collection($supervisor->load('company'));

        return $this->successResponse('Lectura exitosa.', $supervisor);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            # Verificar duplicidad de RUT
            $validate = $this->verifyRut($params['rut'], null, $params['idcompany']);
            if (!$validate->original['status']) return $validate;

            $supervisor = Supervisors::create([
                'rut'       => $params['rut'],
                'name'      => $params['name'],
                'surname'   => $params['surname'],
                'email'     => $params['email'],
                'phone'     => $params['phone'],
                'idcompany' => $params['idcompany']
            ]);
            $supervisor->fresh();

            # Se crea al usuario por el observer

            DB::commit();
            return $this->successResponse('Supervisor creado satisfactoriamente.', $supervisor);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación del supervisor.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar duplicidad de RUT
            $validate = $this->verifyRut($params['rut'] ?? null, $id, $params['idcompany']);
            if (!$validate->original['status']) return $validate;

            # Verificar validez del supervisor
            $validate = $this->verifySupervisor($id);
            if (!$validate->original['status']) return $validate;

            $supervisor = Supervisors::find($id);
            $supervisor->update($params);

            DB::commit();
            return $this->successResponse('Supervisor actualizado satisfactoriamente.', $supervisor);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un supervisor.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del supervisor
            $validate = $this->verifySupervisor($id);
            if (!$validate->original['status']) return $validate;

            $supervisor = Supervisors::find($id);
            $supervisor->update(['status' => 2]);

            # Se desactiva al usurio del supervisor por el observer

            DB::commit();
            return $this->successResponse('Supervisor eliminado satisfactoriamente.', $supervisor);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un supervisor.', $th->getMessage());
        }
    }

    private function verifySupervisor($id)
    {
        $supervisor = Supervisors::activeForID($id)->first();
        if (!$supervisor) return $this->errorResponse('El supervisor seleccionado no esta disponible', 400);

        return $this->successResponse('OK');
    }

    private function verifyRut($rut, $id = null, $idcompany)
    {
        if (!is_null($rut)) {
            $rutExisting = Supervisors::rut($rut, $id)->Company($idcompany)->first();
            // dd($rutExisting);
            if ($rutExisting) return $this->errorResponse('El rut ya se encuentra registrado.', 400);
        }
        return $this->successResponse('OK');
    }
}
