<?php

namespace App\Services;

use App\Models\Product;
use App\Models\Category;
use App\Models\Subcategory;
use App\Models\ProductAttribute;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

class SubcategoryService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $subcategories = Subcategory::where('status', 1)->orderBy('id', 'desc');

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $subcategories->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $subcategories = !empty($withPagination)
            ? $subcategories->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $subcategories->get();

        return $this->successResponse('Lectura exitosa.', $subcategories);
    }

    public function listByCompany($params, $withPagination)
    {
        $subcategories = Subcategory::where('status', 1)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $subcategories->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $subcategories->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $subcategories = !empty($withPagination)
            ? $subcategories->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $subcategories->get();

        return $this->successResponse('Lectura exitosa.', $subcategories);
    }

    public function listDuplicated($idcompany)
    {
        $duplicatedCategories = Subcategory::where('idcompany', $idcompany)
            ->groupBy('name', 'idcompany')
            ->select('name', 'idcompany', \Illuminate\Support\Facades\DB::raw('COUNT(*) as total'))
            ->having('total', '>', 1)
            ->get();

        return $this->successResponse('Subcategorias duplicadas.', $duplicatedCategories);
    }

   
    public function reviewData($params)
    {
        $idcompany = $params['idcompany'];
        $organizedData = [];
    
        // Expresiones regulares para identificar los patrones en la cadena de texto
        $cilindrajePattern = '/\b(\d{1,2}\.\d)\b/';  // Cilindraje: números con decimales como 1.6, 2.0
        $modelPattern = '/([a-zA-Z]+)/';  // Modelos: texto solo con letras (sin versiones o números)
    
        // Usar chunking para procesar los datos en lotes pequeños y no sobrecargar la memoria
        Subcategory::with(['products' => function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany)->with('category');
            }])
            ->where('idcompany', $idcompany)
            ->active()
            ->chunk(100, function ($subfamilies) use (&$organizedData, $cilindrajePattern, $modelPattern) {
                foreach ($subfamilies as $subfamily) {
                   
    
                    foreach ($subfamily->products as $product) {
                        $relatedCategory[] = $product->category ? $product->category->id : '';
                    }
    
                    $subcategoryId = $subfamily->id;
                    $name = $subfamily->name ?: 'Genérico';
                    $subCategoryName = $subfamily->name ?: 'Genérico';
                    $relatedCategory = [];

                    $segments = preg_split('/-/', $name);
    
                    foreach ($segments as $segment) {
                        preg_match_all($modelPattern, $segment, $modelMatches);
                        $models = array_map('trim', $modelMatches[0]);
    
                        preg_match_all($cilindrajePattern, $segment, $cilindrajeMatches);
                        $cilindrajes = array_map('trim', $cilindrajeMatches[0]);
    
                        foreach ($models as $model) {
                            // Si hay cilindrajes, procesamos el modelo con cada cilindraje
                            if (!empty($cilindrajes)) {
                                foreach ($cilindrajes as $cilindraje) {
                                    $normalizedModel = strtolower($model) . ' ' . $cilindraje;
                        
                                    if (!isset($organizedData[$normalizedModel])) {
                                        $organizedData[$normalizedModel] = [
                                            'name' => ucfirst($model) . ' ' . $cilindraje,
                                            'cilindrajes' => [$cilindraje],
                                            'subcategory' => $subCategoryName,
                                            'relatedCategory' => $relatedCategory,
                                            'idreference' => $subcategoryId,
                                            'idparent' => !empty($relatedCategory[0])
                                                ? optional(ProductAttribute::where('idtype', 1)->where('idreference', $relatedCategory[0])->first())->id
                                                : null,
                                            'products' => $subfamily->products
                                        ];
                                    }
                                }
                            } else {
                                // Si no hay cilindrajes, solo guardamos el modelo
                                $normalizedModel = strtolower($model);
                        
                                if (!isset($organizedData[$normalizedModel])) {
                                    $organizedData[$normalizedModel] = [
                                        'name' => ucfirst($model),
                                        'cilindrajes' => [],
                                        'subcategory' => $subCategoryName,
                                        'relatedCategory' => $relatedCategory,
                                        'idreference' => $subcategoryId,
                                        'idparent' => !empty($relatedCategory[0])
                                            ? optional(ProductAttribute::where('idtype', 1)->where('idreference', $relatedCategory[0])->first())->id
                                            : null,
                                        'products' => $subfamily->products
                                    ];
                                }
                            }
                        }
                        
                    }
                }
            });
    
        // Convertir de array asociativo a array indexado
        $result = array_values($organizedData);
    
        return $this->successResponse('Modelos organizados correctamente.', $result);
    }
    
    
    public function listWithProducts($params, $withPagination)
    {
        $subcategories = Subcategory::where('status', 1)
            ->whereHas('products', function ($query) {
                $query->where('status', 1); 
            })
            ->orderBy('id', 'desc');

        $search = isset($params['search']) ? $params['search'] : '';
        if ($search) {
            $subcategories->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }
    
        $subcategories = !empty($withPagination)
            ? $subcategories->paginate($withPagination['perPage'], ['*'], 'page', $withPagination['page'])
            : $subcategories->get();
    
        return $this->successResponse('Lectura exitosa.', $subcategories);
    } 
    
    public function unifyDuplicated($params)
    {
        DB::beginTransaction();
        try {
            $subcategories  = $params['dataSubcategory'];
    
            foreach ($subcategories as $subcategory) {
    
                $name = $subcategory['name'];
                $idcompany = $subcategory['idcompany'];
    
                // Obtener todas las categorías duplicadas con ese nombre
                $duplicatedSubcategories = Subcategory::where('name', $name)
                    ->where('idcompany', $idcompany)
                    ->get();
    
                if ($duplicatedSubcategories->count() > 1) {
                    // Selecciona un ID para mantener activo (el primer ID)
                    $activeSubcategory = $duplicatedSubcategories->first();
                    $activeSubcategoryId = $activeSubcategory->id;
    
                    // Desactiva todas las otras categorías duplicadas excepto la seleccionada
                    Subcategory::where('name', $name)
                        ->where('idcompany', $idcompany)
                        ->where('id', '!=', $activeSubcategoryId)
                        ->update(['status' => 2]); // Cambia el estado a desactivado (status 2)
    
                    // Actualiza los productos que usaban los IDs desactivados
                    Product::whereIn('idsubcategory', $duplicatedSubcategories->pluck('id')->toArray())
                        ->update(['idsubcategory' => $activeSubcategoryId]);
                }
            }
    
            DB::commit();
            return $this->successResponse('Unificación satisfactoria.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('Ocurrió un problema durante la unificación.', $th->getMessage());
        }
    }

    public function listByCategory($params, $withPagination)
    {
        if (!isset($params['idcategory'])) return $this->errorResponse('El idcategory es requerido.', 400);
    
        $category = Category::where('id', $params['idcategory'])->where('status', 1)->first();
        if(!$category) return $this->errorResponse('No se encontró la categoria.', 400);
    
        $idsubcategories = json_decode($category->idsubcategories);
    
        // Verificar si $idsubcategories es nulo o un array vacío
        if (is_null($idsubcategories) || empty($idsubcategories)) {
            return $this->successResponse('Lectura exitosa.', []);
        }
    
        $subcategories = Subcategory::whereIn('id', $idsubcategories)->where('status', 1)->orderBy('id', 'desc');
    
        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $subcategories->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }
    
        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $subcategories->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }
    
        $subcategories = !empty($withPagination)
            ? $subcategories->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $subcategories->get();
    
        return $this->successResponse('Lectura exitosa.', $subcategories);
    }
    

    public function register($params)
    {
        DB::beginTransaction();
        try {
            $nameExisting = Subcategory::where('name', $params['name'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $subcategory = Subcategory::create([
                'name' => $params['name'],
                'description' => $params['description'],
                'idspecificsubcategories' => $params['idspecificsubcategories'],
                'idcompany' => isset($params['idcompany']) ? $params['idcompany'] : null,
            ]);
            $subcategory->save();

            DB::commit();
            return $this->successResponse('Subcategoria creada satisfactoriamente.', $subcategory);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de una Subcategoria.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la subcategoria
            $validate = $this->verifySubcategory($id);
            if (!$validate->original['status']) return $validate;

            $nameExisting = Subcategory::where('name', $params['name'])->where('id','!=', $id)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $subcategory = Subcategory::find($id);
            $subcategory->name = $params['name'];
            $subcategory->description = $params['description'];
            $subcategory->idspecificsubcategories = $params['idspecificsubcategories'];
            $subcategory->save();

            DB::commit();
            return $this->successResponse('Subcategoria actualizada satisfactoriamente.', $subcategory);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de una Subcategoria.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la subcategoria
            $validate = $this->verifySubcategory($id);
            if (!$validate->original['status']) return $validate;

            $subcategory = Subcategory::find($id);
            $subcategory->status = 2;
            $subcategory->save();

            DB::commit();
            return $this->successResponse('Subcategoria eliminada satisfactoriamente.', $subcategory);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de una Subcategoria.', $th->getMessage());
        }
    }

    private function verifySubcategory($idsubcategory)
    {
        $subcategory = Subcategory::where('id', $idsubcategory)->where('status', 1)->first();
        if (!$subcategory) return $this->errorResponse('No se encontró la Subcategoria.', 400);

        return $this->successResponse('OK');
    }
}
