<?php

namespace App\Services;

use App\Http\Resources\StockProductResource;
use App\Models\DetailPriceListProduct;
use App\Models\DetailShoppingCart;
use App\Models\PriceList;
use App\Models\Product;
use App\Models\ShoppingCart;
use App\Models\StockProduct;
use App\Models\Store;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class StockProductService
 * @package App\Services
 */
class StockProductService
{
    use HasResponse;

    public function list($withPagination)
    {
        $stockProduct = StockProduct::stockProductFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc');

        $stockProduct = !empty($withPagination)
            ? $stockProduct->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $stockProduct->get();

        $stockProduct = StockProductResource::collection($stockProduct->load('product', 'warehouse', 'company'));

        return $this->successResponse('Lectura exitosa.', $stockProduct);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            # Verificar duplicidad de sucursal
            $wareHouseExisting = StockProduct::warehouseID($params['idproduct'], $params['idwarehouse'])->active()->first();
            if ($wareHouseExisting) return $this->errorResponse('El producto ya se encuentra registrado a esta bodega.', 400);

            $stockProduct = StockProduct::create([
                'stock_erp'             => $params['stock_erp'],
                'stock_system'          => $params['stock_system'],
                'registration_date'     => $params['registration_date'],
                'last_update'           => $params['registration_date'],
                'idproduct'             => $params['idproduct'],
                'idwarehouse'           => $params['idwarehouse'],
                'idcompany'             => $params['idcompany']
            ]);
            $stockProduct->fresh();

            DB::commit();
            return $this->successResponse('Producto agregado a una bodega satisfactoriamente.', $stockProduct);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la asignacion de un producto a una bodega.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del vendedor
            $validate = $this->verifyProductWarehouse($id, $params['idwarehouse']);
            if (!$validate->original['status']) return $validate;

            $stockProduct = StockProduct::find($id);
            $stockProduct->update($params);

            DB::commit();
            return $this->successResponse('Producto actualizado satisfactoriamente.', $stockProduct);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un producto.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del producto
            $validate = $this->verifyProductWarehouse($id);
            if (!$validate->original['status']) return $validate;

            $stockProduct = StockProduct::find($id);
            $stockProduct->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('Producto eliminado de la bodega satisfactoriamente.', $stockProduct);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un producto de la bodega.', $th->getMessage());
        }
    }

    public function searchProductAvailability($params)
    {
        $stockProduct = StockProduct::stockProductFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc')->get();

        $groupedData = collect($stockProduct)->groupBy('idproduct')->map(function ($items) use ($params) {
            $idproduct = $items->first()['idproduct'];

            $warehouses = $items->map(function ($item) use ($params) {
                $warehouse = Store::where('id', $item['idwarehouse']);
                if ($params['iddistrict']) $warehouse = $warehouse->where('iddistrict', $params['iddistrict']);
                $warehouse = $warehouse->where('status', 1)->first();

                if ($warehouse) {
                    return [
                        'id' => $item['idwarehouse'],
                        'code' => $warehouse->code,
                        'name' => $warehouse->name,
                        'address' => $warehouse->address,
                        'iddistrict' => $warehouse->iddistrict
                    ];
                } else {
                    return null;
                }
            });

            return $warehouses->filter();
        })->values()->first();

        return $this->successResponse('Lectura exitosa.', $groupedData);
    }

    private function verifyProductWarehouse($id, $warehouse = null)
    {
        $stockProduct = StockProduct::activeForID($id)->first();
        if (!$stockProduct) return $this->errorResponse('El producto seleccionado no esta disponible en esta bodega', 400);

        if (isset($warehouse)) {
            $validateWarehouse = StockProduct::branchOfficeID($stockProduct->idproduct, $warehouse, $id)->active()->first();
            if ($validateWarehouse) return $this->errorResponse('La bodega ya se encuentra registrada.', 400);
        }

        return $this->successResponse('OK');
    }

    public function reduceStock($params)
    {
        DB::beginTransaction();
        try {
            # Verificar si se mandó varias bodegas o solo una
            $countWareHouse = count($params['idwarehouse']);

            # Para saber si tengo que recorrer todo el listado o solo quedarme con el primer elemento
            if ($countWareHouse > 1) {
                # Verificar que sea la misma cantidad
                if (count($params['idproducts']) != count($params['idwarehouse'])) {
                    return $this->errorResponse('La cantidad de productos debe ser la misma que la cantidad de bodegas seleccionadas.', 400);
                }
                $checkList = true;
            } else {
                $checkList =  false;
            }

            # Validar fk
            $validate = $this->verifyProductWarehouseCart($params['idproducts'], $params['idwarehouse'], $params['idshopping_cart']);
            if (!$validate->original['status']) return $validate;

            # Reducir stock de productos según bodega(s)
            foreach ($params['idproducts'] as $key => $product) {

                $idWareHouse = $checkList ? $params['idwarehouse'][$key] : $params['idwarehouse'][0];

                # Verificar que el producto tenga relación con la bodega seleccionada
                $stock = StockProduct::where('idproduct', $product)
                    ->where('idwarehouse', $idWareHouse)
                    ->active()->first();

                if (!$stock) return $this->errorResponse('El producto seleccionado no esta en la bodega especificada.', 400);

                # Reducir según la cantidad del carrito

                # Verificar que el producto este dentro del carrito y obtener su cantidad para reducir el stock
                $detailCart = DetailShoppingCart::where('idshopping_cart', $params['idshopping_cart'])
                    ->whereHas('product', function ($q) use ($product) {
                        $q->where('idproduct', $product);
                    })->first();

                $productData = Product::find($product);
                if (!$detailCart) return $this->errorResponse("El producto $productData->name no esta dentro del carrito seleccionado.", 400);

                $amountReduce = $detailCart->quantity;

                # Reducir del stock
                $stock->decrement('stock_system', $amountReduce);
            }

            DB::commit();
            return $this->successResponse('Reducción de stock completa con éxito.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la reducción de stock de un producto.', $th->getMessage());
        }
    }

    private function verifyProductWarehouseCart($products, $warehouse, $cart)
    {
        $productsCount = Product::whereIn('id', $products)->active()->count();
        if ($productsCount != count($products)) return $this->errorResponse('Verifique sus productos seleccionados.', 400);

        $warehouseCount = Store::whereIn('id', $warehouse)->active()->count();
        if ($warehouseCount != count($warehouse)) return $this->errorResponse('Verifique sus bodegas seleccionadas.', 400);

        $cart = ShoppingCart::where('id', $cart)->active()->first();
        if (!$cart) return $this->errorResponse('El carrito de compras seleccionado es inválido.', 400);

        return $this->successResponse('OK');
    }
}
