<?php

namespace App\Services;

use App\Models\SpecificUse;
use App\Models\UsageAttribute;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

class SpecificUsesService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $attributes = SpecificUse::where('status', 1)->orderBy('id', 'desc');

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $attributes->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%")
                ->orWhere('code', 'like', "%$search%");
            });
        }

        $attributes = !empty($withPagination)
            ? $attributes->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $attributes->get();

        return $this->successResponse('Lectura exitosa.', $attributes);
    }

    public function listByCompany($params, $withPagination)
    {
        $attributes = SpecificUse::where('status', 1)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $attributes->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $attributes->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%")
                ->orWhere('code', 'like', "%$search%");
            });
        }

        $attributes = !empty($withPagination)
            ? $attributes->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $attributes->get();

        return $this->successResponse('Lectura exitosa.', $attributes);
    }

    public function listByUse($params, $withPagination)
    {
        if (!isset($params['iduse'])) return $this->errorResponse('El iduse es requerido.', 400);

        $use = UsageAttribute::where('id', $params['iduse'])->where('status', 1)->first();
        if(!$use) return $this->errorResponse('No se encontró el atributo de uso.', 400);

        if($use->idspecificuses == null || $use->idspecificuses == "[]") return $this->errorResponse('Aún no se ha seleccionado usos especificas.', 400);

        $idspecificuses = json_decode($use->idspecificuses);

        $attributes = SpecificUse::whereIn('id', $idspecificuses)->where('status', 1)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $attributes->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $attributes->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%")
                ->orWhere('code', 'like', "%$search%");
            });
        }

        $attributes = !empty($withPagination)
            ? $attributes->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $attributes->get();

        return $this->successResponse('Lectura exitosa.', $attributes);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            
            $nameExisting = SpecificUse::where('name', $params['name'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $codeExisting = SpecificUse::where('code', $params['code'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($codeExisting) return $this->errorResponse('El código ya se encuentra registrado.', 400);

            $attribute = SpecificUse::create([
                'code' => $params['code'],
                'name' => $params['name'],
                'description' => $params['description'],
                'idcompany' => isset($params['idcompany']) ? $params['idcompany'] : null,
            ]);
            $attribute->save();

            DB::commit();
            return $this->successResponse('Uso specífico creado satisfactoriamente.', $attribute);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un uso específico.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la atributo
            $validate = $this->verifyAttribute($id);
            if (!$validate->original['status']) return $validate;

            $nameExisting = SpecificUse::where('name', $params['name'])->where('id','!=', $id)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $codeExisting = SpecificUse::where('code', $params['code'])->where('id','!=', $id)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($codeExisting) return $this->errorResponse('El código ya se encuentra registrado.', 400);

            $attribute = SpecificUse::find($id);
            $attribute->code = $params['code'];
            $attribute->name = $params['name'];
            $attribute->description = $params['description'];
            $attribute->save();

            DB::commit();
            return $this->successResponse('Uso específico actualizado satisfactoriamente.', $attribute);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un uso específico.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la atributo
            $validate = $this->verifyAttribute($id);
            if (!$validate->original['status']) return $validate;

            $attribute = SpecificUse::find($id);
            $attribute->status = 2;
            $attribute->save();

            DB::commit();
            return $this->successResponse('Uso específico eliminado satisfactoriamente.', $attribute);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un uso específico.', $th->getMessage());
        }
    }

    private function verifyAttribute($idattribute)
    {
        $attribute = SpecificUse::where('id', $idattribute)->where('status', 1)->first();
        if (!$attribute) return $this->errorResponse('No se encontró el uso específico.', 400);

        return $this->successResponse('OK');
    }
}
