<?php

namespace App\Services;

use App\Models\SpecificSubcategory;
use App\Models\Subcategory;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

class SpecificSubcategoryService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $subcategories = SpecificSubcategory::where('status', 1)->orderBy('id', 'desc');

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $subcategories->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $subcategories = !empty($withPagination)
            ? $subcategories->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $subcategories->get();

        return $this->successResponse('Lectura exitosa.', $subcategories);
    }

    public function listByCompany($params, $withPagination)
    {
        $subcategories = SpecificSubcategory::where('status', 1)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $subcategories->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $subcategories->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $subcategories = !empty($withPagination)
            ? $subcategories->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $subcategories->get();

        return $this->successResponse('Lectura exitosa.', $subcategories);
    }

    public function listBySubcategory($params, $withPagination)
    {
        if (!isset($params['idsubcategory'])) return $this->errorResponse('El idsubcategory es requerido.', 400);

        $subcategory = Subcategory::where('id', $params['idsubcategory'])->where('status', 1)->first();
        if(!$subcategory) return $this->errorResponse('No se encontró la subcategoria.', 400);

        if($subcategory->idspecificsubcategories == null || $subcategory->idspecificsubcategories == "[]") return $this->errorResponse('Aún no se ha seleccionado subcategorias especificas.', 400);

        $idspecificsubcategories = json_decode($subcategory->idspecificsubcategories);

        $subcategories = SpecificSubcategory::whereIn('id', $idspecificsubcategories)->where('status', 1)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $subcategories->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $subcategories->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%");
            });
        }

        $subcategories = !empty($withPagination)
            ? $subcategories->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $subcategories->get();

        return $this->successResponse('Lectura exitosa.', $subcategories);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            
            $nameExisting = SpecificSubcategory::where('name', $params['name'])->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $subcategory = SpecificSubcategory::create([
                'name' => $params['name'],
                'description' => $params['description'],
                'idcompany' => isset($params['idcompany']) ? $params['idcompany'] : null,
            ]);
            $subcategory->save();

            DB::commit();
            return $this->successResponse('Subcategoria específica creada satisfactoriamente.', $subcategory);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de una Subcategoria específica.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la subcategoria
            $validate = $this->verifySubcategory($id);
            if (!$validate->original['status']) return $validate;

            $nameExisting = SpecificSubcategory::where('name', $params['name'])->where('id','!=', $id)->where('idcompany', $params['idcompany'])->where('status', 1)->first();
            if ($nameExisting) return $this->errorResponse('El nombre ya se encuentra registrado.', 400);

            $subcategory = SpecificSubcategory::find($id);
            $subcategory->name = $params['name'];
            $subcategory->description = $params['description'];
            $subcategory->save();

            DB::commit();
            return $this->successResponse('Subcategoria específica actualizada satisfactoriamente.', $subcategory);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de una Subcategoria específica.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de la subcategoria
            $validate = $this->verifySubcategory($id);
            if (!$validate->original['status']) return $validate;

            $subcategory = SpecificSubcategory::find($id);
            $subcategory->status = 2;
            $subcategory->save();

            DB::commit();
            return $this->successResponse('Subcategoria específica eliminada satisfactoriamente.', $subcategory);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de una Subcategoria específica.', $th->getMessage());
        }
    }

    private function verifySubcategory($idsubcategory)
    {
        $subcategory = SpecificSubcategory::where('id', $idsubcategory)->where('status', 1)->first();
        if (!$subcategory) return $this->errorResponse('No se encontró la Subcategoria específica.', 400);

        return $this->successResponse('OK');
    }
}
