<?php

namespace App\Services;

use App\Http\Resources\DistrictResource;
use App\Models\BranchOffice;
use App\Models\BusinessArea;
use App\Models\Category;
use App\Models\DetailPriceListProduct;
use App\Models\LogApiERP;
use App\Models\Product;
use App\Models\SoftnetWayToPay;
use App\Models\Store;
use App\Models\UnitMeasure;
use App\Models\UserERP;
use App\Models\Ticket;
use App\Models\DetailTicket;
use App\Models\District;
use App\Models\Province;
use App\Models\Department;
use App\Models\Country;
use App\Models\SpecificSubcategory;
use App\Models\StockProduct;
use App\Models\WayToPay;
use App\Models\WayToPayForCompany;
use App\Models\Subcategory;
use App\Models\Supplier;
use App\Models\ClientAddress;
use App\Traits\HasResponse;
use Carbon\Carbon;
use GuzzleHttp\Client;
use GuzzleHttp\Psr7\Request;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\Date;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Tymon\JWTAuth\Facades\JWTAuth;
use Tymon\JWTAuth\Exceptions\JWTException;

class SoftnetService
{
    use HasResponse;

    public function decryptTokenSoftnet($token)
    {
        return Crypt::decryptString($token);
    }

    public function getLoginSoftnet($userErp)
    {
        try {
            $response = Http::withHeaders([
                'Access-Control-Allow-Origin' => '*',
                'Content-Type' => 'application/x-www-form-urlencoded',
            ])->post('http://api.softnet.cl/login', [
                "username" => $userErp->user,
                "password" => $userErp->password,
                "rut" => $userErp->rut
            ]);

            $query = $response->json();

            return $query['token'];
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getTokenSoftnet($userErp)
    {
        try {
            $response = Http::withHeaders([
                'Access-Control-Allow-Origin' => '*',
                'Content-Type' => 'application/x-www-form-urlencoded',
            ])->post('http://api.softnet.cl/login', [
                "username" => $userErp->user,
                "password" => Crypt::decryptString($userErp->encrypted_password),
                "rut" => $userErp->rut
            ]);

            $query = $response->json();

            return $query['token'];
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function GetApiSoftnet($endpoint, $params)
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $params['token_erp'] = $token;

        $url = 'http://api2.softnet.cl/' . $endpoint;

        $response = Http::withHeaders([
            'Access-Control-Allow-Origin' => '*',
            'Content-Type' => 'application/x-www-form-urlencoded',
            'token' => $params['token_erp']
        ])->get($url);

        $response = $response->json();

        //registro log
        DB::beginTransaction();
        try {
            LogApiERP::create([
                'endpoint' => $endpoint,
                'metodo' => 'GET',
                'entrada' => '',
                'respuesta' => strlen(json_encode($response)) < 65535 ? json_encode($response) : ''
            ]);
            DB::commit();
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }

        return $response;
    }

    public function getAreaNegocio($params)
    {
        try {
            $endpoint = 'areaNegocio';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getFormaPago($params)
    {
        try {
            $endpoint = 'formaPago';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getVendedor($params)
    {
        try {
            $endpoint = 'vendedores';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getBodega($params)
    {
        try {
            $endpoint = 'bodega';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getCategoria($params)
    {
        try {
            $endpoint = 'familia';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getSubcategoria($params)
    {
        try {
            $endpoint = 'subfamilia';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getLinea($params)
    {
        try {
            $endpoint = 'linea_productos';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getUsuario($params)
    {
        try {
            $endpoint = 'datoUsuario';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getCliente($params)
    {
        try {
            $endpoint = 'cliente';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getClienteAddress($params)
    {
        try {
            $endpoint = 'clienteDireccion';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }


    public function getProveedor($params)
    {
        try {
            $endpoint = 'proveedor';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getSucursal($params)
    {
        try {
            $endpoint = 'sucursal';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function EmiteFactura($params, $dte, $idticket, $tipoDocumento)
    {
        try {

            $url = '';
            $endpoint = '';
            switch ($tipoDocumento) {
                case 1:
                    $url = 'http://api.softnet.cl/boletaLite';
                    $endpoint = 'boleta';
                    break;

                case 2:
                    $url = 'http://api.softnet.cl/facturaLite';
                    $endpoint = 'factura';
                    break;

                case 3:
                    $url = 'http://api.softnet.cl/notaCreditoLite';
                    $endpoint = 'nota_credito';
                    break;

                case 4:
                    $url = 'http://api.softnet.cl/notaDebitoLite';
                    $endpoint = 'nota_debito';
                    break;

                case 5:
                    $url = 'http://api.softnet.cl/cotizacionLite';
                    $endpoint = 'cotizacion';
                    break;

                default:
                    break;
            }

            $data = json_encode($dte);

            $response = Http::withBody($data, 'application/json')
                ->withHeaders([
                    'Access-Control-Allow-Origin' => '*',
                    'Content-Type' => 'application/x-www-form-urlencoded',
                    'token' => $this->decryptTokenSoftnet($params['token_erp'])
                ])
                ->post($url);

            $response = $response->json();

            LogApiERP::create([
                'idticket' => $idticket, //agregar
                'endpoint' => $endpoint,
                'metodo' => 'POST SFT SERV',
                'entrada' => $data,
                'respuesta' => json_encode($response)
            ]);

            return $response;
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getFormaPagoERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $warehouses = $this->getFormaPago($request);
        if (!$warehouses->original['status']) return $warehouses;

        $warehouses = $warehouses->original['data']['detail'];

        return $this->successResponse('OK', $warehouses);
    }

    public function getProductos($params)
    {
        try {
            $endpoint = 'producto';
            $response = $this->GetApiSoftnet($endpoint, $params);

            return $this->successResponse('Lectura exitosa', $response);
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function consultaDTE($params, $TipoDTE, $folio, $rut)
    {
        try {
            $url = 'http://api.softnet.cl/consultaDTE';
            $dte = (object)[
                'tipo'  => $TipoDTE,
                'folio' => $folio,
                'rut'   => $rut
            ];
            $data = json_encode($dte);

            $response = Http::withBody($data, 'application/json')
                ->withHeaders([
                    'Access-Control-Allow-Origin' => '*',
                    'Content-Type'  => 'application/json',
                    'token'         => $this->decryptTokenSoftnet($params['token_erp'])
                ])
                ->post($url);

            $response = $response->json();

            LogApiERP::create([
                'endpoint' => 'consulta DTE',
                'metodo' => 'POST SFT SERV',
                'entrada' => $data,
                'respuesta' => json_encode($response)
            ]);

            return $response;
        } catch (\Throwable $th) {
            throw $th;
        }
    }

    public function getWareHouseERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $warehouses = $this->getBodega($request);
        if (!$warehouses->original['status']) return $warehouses;

        $warehouses = $warehouses->original['data']['detail'];

        return $this->successResponse('OK', $warehouses);
    }

    public function getBusinessAreaERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $businessAreas = $this->getAreaNegocio($request);
        if (!$businessAreas->original['status']) return $businessAreas;

        $businessAreas = $businessAreas->original['data']['detail'];

        return $this->successResponse('OK', $businessAreas);
    }

    public function getProductsERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $products = $this->getProductos($request);
        if (!$products->original['status']) return $products;

        $products = $products->original['data']['detail'];

        return $this->successResponse('OK', $products);
    }

    public function getCategoryERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $categories = $this->getCategoria($request);
        if (!$categories->original['status']) return $categories;

        $categories = $categories->original['data']['detail'];

        return $this->successResponse('OK', $categories);
    }

    public function getSubcategoryERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $subcategories = $this->getSubcategoria($request);
        if (!$subcategories->original['status']) return $subcategories;

        $subcategories = $subcategories->original['data']['detail'];

        return $this->successResponse('OK', $subcategories);
    }

    public function getSpecificSubcategoryERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $specificsubcategory = $this->getLinea($request);
        if (!$specificsubcategory->original['status']) return $specificsubcategory;

        $specificsubcategory = $specificsubcategory->original['data']['detail'];

        return $this->successResponse('OK', $specificsubcategory);
    }

    public function getUserERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $users = $this->getUsuario($request);
        if (!$users->original['status']) return $users;

        $users = $users->original['data']['detail'];

        return $this->successResponse('OK', $users);
    }

    public function getCustomerERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $customers = $this->getCliente($request);
        if (!$customers->original['status']) return $customers;

        $customers = $customers->original['data']['detail'];

        return $this->successResponse('OK', $customers);
    }

    public function getCustomerAddressERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $customers = $this->getClienteAddress($request);
        if (!$customers->original['status']) return $customers;

        $customers = $customers->original['data']['detail'];

        return $this->successResponse('OK', $customers);
    }

    public function getSupplierERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $suppliers = $this->getProveedor($request);
        if (!$suppliers->original['status']) return $suppliers;

        $suppliers = $suppliers->original['data']['detail'];

        return $this->successResponse('OK', $suppliers);
    }

    public function getBranchOfficeERP()
    {
        $userErp = UserERP::where('iduser', Auth::user()->id)->first();
        if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

        $token = $this->getTokenSoftnet($userErp);
        $request['token_erp'] = $token;

        $branch = $this->getSucursal($request);
        if (!$branch->original['status']) return $branch;

        $branch = $branch->original['data']['detail'];

        return $this->successResponse('OK', $branch);
    }

    public function importFormaPago()
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;

            $waypays = $this->getFormaPago($request);
            if (!$waypays->original['status']) return $waypays;

            $waypays = $waypays->original['data']['detail'];

            $dataWaypays = [];
            $dataUpdateWayPays = false;
            foreach ($waypays as $waypay) {
                $verifyWaypay = WayToPay::where('iderp', (int)$waypay['id'])->first();
                if (!$verifyWaypay) {
                    $dataWaypays[] = [
                        'iderp'         => $waypay['id'],
                        'name'          => $waypay['nombre'],
                        'agenda'        => $waypay['agenda'],
                        'SII_fmapago'   => $waypay['SII_fmapago'],
                        'created_at'    => $now,
                        'updated_at'    => $now
                    ];
                } else {
                    $verifyWaypay->update([
                        'name'          => $waypay['nombre'],
                        'agenda'        => $waypay['agenda'],
                        'SII_fmapago'   => $waypay['SII_fmapago'],
                        'updated_at'    => $now
                    ]);
                    $dataUpdateWayPays = true;
                }
            }

            if (!empty($dataWaypays) && !$dataUpdateWayPays) {
                WayToPay::insert($dataWaypays);

                $idsWaypay = [];
                foreach ($dataWaypays as $dataWaypay) {
                    $waypay = WayToPay::where('iderp', $dataWaypay['iderp'])->active()->first();

                    $idsWaypay[] = [
                        'idway_to_pay'  => $waypay->id,
                        'idcompany'     => request('idcompany')
                    ];
                }

                WayToPayForCompany::insert($idsWaypay);

                DB::commit();
                return $this->successResponse('Formas de pago importadas satisfactoriamente.');
            } elseif ($dataUpdateWayPays) {
                DB::commit();
                return $this->successResponse('Formas de pago actualizadas satisfactoriamente.');
            }

            DB::commit();
            return $this->successResponse('No hay formas de pago para importar.');
        } catch (\Throwable $th) {
            return $this->externalError('durante la importación de forma de pagos.', $th->getMessage());
        }
    }

    public function importSelectFormaPago($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $waypays = $params['dataWaypay'];

            $dataWaypays = [];
            foreach ($waypays as $waypay) {
                $verifyWaypay = WayToPay::where('iderp', (int)$waypay['id'])->active()->first();
                if (!$verifyWaypay) {
                    $dataWaypays[] = [
                        'iderp'         => $waypay['id'],
                        'name'          => $waypay['nombre'],
                        'agenda'        => $waypay['agenda'],
                        'SII_fmapago'   => $waypay['SII_fmapago'],
                        'created_at'    => $now,
                        'updated_at'    => $now
                    ];
                }
            }

            if (!empty($dataWaypays)) {
                WayToPay::insert($dataWaypays);

                $idsWaypay = [];
                foreach ($dataWaypays as $dataWaypay) {
                    $waypay = WayToPay::where('iderp', $dataWaypay['iderp'])->active()->first();

                    $idsWaypay[] = [
                        'idway_to_pay'  => $waypay->id,
                        'idcompany'     => request('idcompany')
                    ];
                }

                WayToPayForCompany::insert($idsWaypay);

                DB::commit();
                return $this->successResponse('Formas de pago importadas satisfactoriamente.');
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de forma de pagos.', $th->getMessage());
        }
    }

    public function importWarehouse($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;

            $warehouses = $this->getBodega($request);
            if (!$warehouses->original['status']) return $warehouses;

            $warehouses = $warehouses->original['data']['detail'];

            $dataWarehouses = [];
            foreach ($warehouses as $warehouse) {
                $verifyWarehouse = Store::where('iderp', (int)$warehouse['id'])->active()->first();
                if (!$verifyWarehouse) {
                    $dataWarehouses[] = [
                        'code'              => "ERP-" . $warehouse['id'],
                        'iderp'             => (int)$warehouse['id'],
                        'name'              => $warehouse['nombre'],
                        'status'            => $warehouse['estado'] === 0 ? "2" : "1",
                        // 'idbranchoffice',
                        // 'iddistrict',
                        // 'idprovince',
                        // 'iddepartment',
                        // 'idcountry',
                        'idcompany' => $params['idcompany'],
                        'created_at' => $now,
                        'updated_at' => $now
                    ];
                }
            }

            if (!empty($dataWarehouses)) {
                Store::insert($dataWarehouses);

                DB::commit();
                return $this->successResponse('Bodegas importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay bodegas para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de bodegas.', $th->getMessage());
        }
    }

    public function importSelectWarehouse($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $warehouses = $params['dataWarehouse'];

            $dataWarehouses = [];
            foreach ($warehouses as $warehouse) {
                $verifyWarehouse = Store::where('iderp', (int)$warehouse['id'])->active()->first();
                if (!$verifyWarehouse) {
                    $dataWarehouses[] = [
                        'code'              => "ERP-" . $warehouse['id'],
                        'iderp'             => (int)$warehouse['id'],
                        'name'              => $warehouse['nombre'],
                        // 'idbranchoffice',
                        // 'iddistrict',
                        // 'idprovince',
                        // 'iddepartment',
                        // 'idcountry',
                        'idcompany' => $params['idcompany'],
                        'created_at' => $now,
                        'updated_at' => $now
                    ];
                }
            }

            if (!empty($dataWarehouses)) {
                Store::insert($dataWarehouses);

                DB::commit();
                return $this->successResponse('Bodegas importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay bodegas para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de bodegas.', $th->getMessage());
        }
    }

    public function importBusinessArea($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;

            $businessAreas = $this->getAreaNegocio($request);
            if (!$businessAreas->original['status']) return $businessAreas;

            $businessAreas = $businessAreas->original['data']['detail'];

            $dataBusinessAreas = [];
            foreach ($businessAreas as $businessArea) {
                $verifyBusinessAreas = BusinessArea::where('iderp', (int)$businessArea['id'])->active()->first();
                if (!$verifyBusinessAreas) {
                    $dataBusinessAreas[] = [
                        'iderp'             => (int)$businessArea['id'],
                        'name'              => $businessArea['nombre'],
                        // 'idbranchoffice',
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataBusinessAreas)) {
                BusinessArea::insert($dataBusinessAreas);

                DB::commit();
                return $this->successResponse('Areas de negocios importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay areas de negocio para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de bodegas.', $th->getMessage());
        }
    }

    public function importSelectBusinessArea($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $businessAreas = $params['dataBusinessAreas'];

            $dataBusinessAreas = [];
            foreach ($businessAreas as $businessArea) {
                $verifyBusinessAreas = BusinessArea::where('iderp', (int)$businessArea['id'])->active()->first();
                if (!$verifyBusinessAreas) {
                    $dataBusinessAreas[] = [
                        'iderp'             => (int)$businessArea['id'],
                        'name'              => $businessArea['nombre'],
                        // 'idbranchoffice',
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataBusinessAreas)) {
                BusinessArea::insert($dataBusinessAreas);

                DB::commit();
                return $this->successResponse('Areas de negocios importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay areas de negocio para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de areas de negocio.', $th->getMessage());
        }
    }

    public function importProductsOld($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;

            $products = $this->getProductos($request);
            if (!$products->original['status']) return $products;

            $products = $products->original['data']['detail'];

            $dataProducts = [];
            $dataProductsLimit = array_slice($products, 0, 1000);
           
            foreach ($dataProductsLimit as $product) {
                
                $verifyProducts = Product::where('code_erp', $product['codigo'])->active()->first();
                if (!$verifyProducts && $product['estado'] == 1) {
                    $unitMeasure = UnitMeasure::where(
                        fn ($query) => $query->where('name', $product['unidad_medida'])
                            ->orWhere('abbreviation', $product['unidad_medida'])
                    )->active()->first();

                    $subcategory = Subcategory::where('iderp', (int)$product['linea_id'])->active()->first();

                    $category = Category::where('iderp', (int)$product['sub_familia_id'])->active()->first();

                    $specificsubcategory = SpecificSubcategory::where('iderp', (int)$product['familia_id'])->active()->first();

                    $dataProducts[] = [
                        // "codigo": "00000",
                        // "codigo_proveedor": "V113181F50042",
                        // "codigo_barra": "V113181F50042",
                        // "nombre": "Sunmi V1 demo",
                        // "costo_maestro": 5000,
                        // "precio": "10115",
                        // "exento": "",
                        // "precio_oferta": "",
                        // "fecha_oferta": "",
                        // "web_estado": 0,
                        // "estado": 1,
                        // "unidad_medida": "UNID",
                        // "tipo_producto_id": 6,
                        // "unidad_medida_id": "1",
                        // "peso": "10",
                        // "altura": 20,
                        // "ancho": 30,
                        // "longitud": 40,
                        // "linea_id": 1410,
                        // "familia_id": 0,
                        // "sub_familia_id": 0,
                        // "marca_id": 0,
                        // "stock": 177791

                        'code_erp'          => $product['codigo'],
                        'code_system'       => $product['codigo'],
                        'barcode'           => $product['codigo_barra'],
                        'name'              => $product['nombre'],
                        'cost'              => (int)$product['costo_maestro'],
                        'price'             => (int)$product['precio'],
                        // 'margin'            => ,
                        'offerprice'        => (int)$product['precio_oferta'],

                        // 'idcointype',
                        'idcategory'        => $category->id ?? null,
                        'idsubcategory'     => $subcategory->id ?? null,
                        'idspecificsubcategory' => $specificsubcategory->id ?? null,
                        // 'idtype',
                        // 'idsupplier',
                        'idunitmeasure'    => $unitMeasure->id ?? null,
                        // 'idbusinessarea',
                        // 'idclassification',
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataProducts)) {
                Product::insert($dataProducts);

                foreach($dataProducts as $key => $product) {
                    if(!is_null($product['idcategory'])){
                        $category = Category::where('id', $product['idcategory'])->active()->first();
                        $category->update(['quantity_products' => $category->quantity_products + 1]);
                    }
                }

                DB::commit();
                return $this->successResponse('productos importadas satisfactoriamente.');
            }


            return $this->successResponse('No hay productos para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de productos.', $th->getMessage());
        }
    }

    public function importProducts($params)
    {
        // Permitir que este script tenga ejecución indefinida para evitar tiempos de espera
        set_time_limit(0);
    
        DB::beginTransaction();
        try {
    
            $now = Carbon::now();
            $importedCount = 0;
    
            // Inicializamos los datos en memoria que se usarán repetidamente
            $unitMeasures = UnitMeasure::active()->get()->keyBy('name')->toArray();
            $unitMeasuresByAbbreviation = UnitMeasure::active()->get()->keyBy('abbreviation')->toArray();
            $subcategories = Subcategory::active()->get()->keyBy('iderp')->toArray();
            $categories = Category::active()->get()->keyBy('iderp')->toArray();
            $specificSubcategories = SpecificSubcategory::active()->get()->keyBy('iderp')->toArray();
    
            // Usamos chunk() para procesar los registros en lotes
            DB::connection('mysql_external')->table('productos')
                ->select('CODIGO_PRODUCTO as codigo', 'CODIGO_BARRA as codigo_barra',
                    'DESCRIPCION_PRODUCTO as nombre', 'ID_UMEDIDA as unidad_medida',
                    'ID_FAMILIAPRODUCTO as sub_familia_id', 'ID_FAMILIAFAMILIA as familia_id',
                    'ID_LINEAPRODUCTO as linea_id', 'COSTO_REPOSICION as costo_maestro',
                    'VALOR_PRODUCTO as precio', 'PRECIO_OFERTA as precio_oferta')
                ->where('ID_EMPRESA', $params['rut_company'])
                ->where('ESTADO_PRODUCTO', 1)
                ->orderBy('CODIGO_PRODUCTO')
                ->chunk(1000, function ($products) use ($params, $now, &$importedCount, $unitMeasures, $unitMeasuresByAbbreviation, $subcategories, $categories, $specificSubcategories) {
    
                    $dataProducts = [];
    
                    // Paso 1: Obtener todos los códigos ya existentes en la base de datos
                    $existingProducts = Product::whereIn('code_erp', $products->pluck('codigo'))->pluck('code_erp')->toArray();
    
                    // Iterar sobre los productos
                    foreach ($products as $product) {
                        // Verificar si el producto ya existe en la base de datos (en memoria)
                        if (in_array($product->codigo, $existingProducts)) {
                            continue; // Si el producto ya existe, saltarlo
                        }
    
                        // Buscar unidad de medida en memoria
                        $unitMeasure = $unitMeasures[$product->unidad_medida] ?? $unitMeasuresByAbbreviation[$product->unidad_medida] ?? null;
    
                        // Buscar categoría, subcategoría y subcategoría específica en memoria
                        $subcategory = $subcategories[$product->sub_familia_id] ?? null;
                        $category = $categories[$product->familia_id] ?? null;
                        $specificsubcategory = $specificSubcategories[$product->linea_id] ?? null;
    
                        // Preparar datos del producto para insertar
                        $dataProducts[] = [
                            'code_erp'          => $product->codigo,
                            'code_system'       => $product->codigo,
                            'barcode'           => $product->codigo_barra,
                            'name'              => $product->nombre,
                            'cost'              => (int)$product->costo_maestro,
                            'price'             => (int)$product->precio,
                            'offerprice'        => (int)$product->precio_oferta,
                            'idcategory'        => $category['id'] ?? null,
                            'idsubcategory'     => $subcategory['id'] ?? null,
                            'idspecificsubcategory' => $specificsubcategory['id'] ?? null,
                            'idunitmeasure'     => $unitMeasure['id'] ?? null,
                            'idcompany'         => $params['idcompany'],
                            'created_at'        => $now,
                            'updated_at'        => $now
                        ];
    
                        // Incrementar el contador de productos importados
                        $importedCount++;
                    }
    
                    // Si hay productos para insertar, realizar la inserción
                    if (!empty($dataProducts)) {
                        Product::insert($dataProducts);
    
                        // Actualizar la cantidad de productos en las categorías
                        foreach ($dataProducts as $product) {
                            if (!is_null($product['idcategory'])) {
                                $category = Category::where('id', $product['idcategory'])->active()->first();
                                $category->update(['quantity_products' => $category->quantity_products + 1]);
                            }
                        }
                    }
                });
    
            // Si no hay productos importados
            if ($importedCount === 0) {
                DB::rollBack();
                return $this->successResponse('No hay productos para importar.');
            }
    
            DB::commit();
            return $this->successResponse("Se importaron $importedCount productos satisfactoriamente.");
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('Error durante la importación de productos.', $th->getMessage());
        }
    }
    

    public function importSelectProducts($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $products = $params['dataProducts'];

            $dataProducts = [];
            foreach ($products as $product) {
                $verifyProducts = Product::where('code_erp', $product['codigo'])->active()->first();
                if (!$verifyProducts) {
                    $unitMeasure = UnitMeasure::where(
                        fn ($query) => $query->where('name', $product['unidad_medida'])
                            ->orWhere('abbreviation', $product['unidad_medida'])
                    )->active()->first();

                    $subcategory = Subcategory::where('iderp', (int)$product['linea_id'])->active()->first();

                    $category = Category::where('iderp', (int)$product['sub_familia_id'])->active()->first();

                    $specificsubcategory = SpecificSubcategory::where('iderp', (int)$product['familia_id'])->active()->first();

                    $dataProducts[] = [
                        'code_erp'          => $product['codigo'],
                        'code_system'       => $product['codigo'],
                        'barcode'           => $product['codigo_barra'],
                        'name'              => $product['nombre'],
                        'cost'              => (int)$product['costo_maestro'],
                        'price'             => (int)$product['precio'],
                        // 'margin'            => ,
                        'offerprice'        => (int)$product['precio_oferta'],

                        // 'idcointype',
                        'idcategory'        => $category->id ?? null,
                        'idsubcategory'     => $subcategory->id ?? null,
                        'idspecificsubcategory' => $specificsubcategory->id ?? null,
                        // 'idtype',
                        // 'idsupplier',
                        'idunitmeasure'    => $unitMeasure->id ?? null,
                        // 'idbusinessarea',
                        // 'idclassification',
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataProducts)) {
                Product::insert($dataProducts);

                DB::commit();
                return $this->successResponse('productos importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay productos para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de productos.', $th->getMessage());
        }
    }

    public function importCategory($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;

            $categories = $this->getCategoria($request);
            if (!$categories->original['status']) return $categories;

            $categories = $categories->original['data']['detail'];

            $dataCategory = [];
            foreach ($categories as $category) {
                $verifyCategory = Category::where('iderp', (int)$category['id'])->active()->first();
                if (!$verifyCategory) {
                    $dataCategory[] = [
                        'iderp'             => (int)$category['id'],
                        'name'              => $category['nombre'],
                        'description'       => $category['nombre'],
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataCategory)) {
                Category::insert($dataCategory);

                DB::commit();
                return $this->successResponse('Categorias importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay categoria para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de bodegas.', $th->getMessage());
        }
    }

    public function importSelectCategory($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $categories = $params['dataCategory'];

            $dataCategory = [];
            foreach ($categories as $category) {
                $verifyCategory = Category::where('iderp', (int)$category['id'])->active()->first();
                if (!$verifyCategory) {
                    $dataCategory[] = [
                        'iderp'             => (int)$category['id'],
                        'name'              => $category['nombre'],
                        'description'       => $category['nombre'],
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataCategory)) {
                Category::insert($dataCategory);

                DB::commit();
                return $this->successResponse('Categorias importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay categoria para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de categorias.', $th->getMessage());
        }
    }

    public function importSubcategory($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;

            $subcategories = $this->getSubcategoria($request);
            if (!$subcategories->original['status']) return $subcategories;

            $subcategories = $subcategories->original['data']['detail'];

            $dataSubcategory = [];
            foreach ($subcategories as $subcategory) {
                $verifySubcategory = Subcategory::where('iderp', (int)$subcategory['id'])->active()->first();
                if (!$verifySubcategory) {
                    $dataSubcategory[] = [
                        'iderp'             => (int)$subcategory['id'],
                        'name'              => $subcategory['descripcion'],
                        'description'       => $subcategory['descripcion'],
                        // 'idcategory'        => $subcategory['idfamilia'] ?? null,
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataSubcategory)) {
                Subcategory::insert($dataSubcategory);

                DB::commit();
                return $this->successResponse('Subcategorias importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay subcategoria para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de subcategorias.', $th->getMessage());
        }
    }

    public function importSelectSubcategory($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $subcategories = $params['dataSubcategory'];

            $dataSubcategory = [];
            foreach ($subcategories as $subcategory) {
                $verifySubcategory = Subcategory::where('iderp', (int)$subcategory['id'])->active()->first();
                if (!$verifySubcategory) {
                    $dataSubcategory[] = [
                        'iderp'             => (int)$subcategory['id'],
                        'name'              => $subcategory['nombre_subFamilia'],
                        'description'       => $subcategory['descripcion'],
                        'idcategory'        => $subcategory['idfamilia'],
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataSubcategory)) {
                Subcategory::insert($dataSubcategory);

                DB::commit();
                return $this->successResponse('Subcategorias importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay subcategoria para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de subcategorias.', $th->getMessage());
        }
    }

    public function importSpecificsubcategory($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;

            $specificsubcategories = $this->getSpecificSubcategoryERP($request);
            if (!$specificsubcategories->original['status']) return $specificsubcategories;

            $specificsubcategories = $specificsubcategories->original['data']['detail'];

            $dataSpecificsubcategory = [];
            foreach ($specificsubcategories as $specificsubcategory) {
                $verifySpecificsubcategory = SpecificSubcategory::where('iderp', (int)$specificsubcategory['id'])->active()->first();
                if (!$verifySpecificsubcategory) {
                    $dataSpecificsubcategory[] = [
                        'iderp'             => (int)$specificsubcategory['id'],
                        'name'              => $specificsubcategory['nombre'],
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataSpecificsubcategory)) {
                SpecificSubcategory::insert($dataSpecificsubcategory);

                DB::commit();
                return $this->successResponse('Subcategorias especificas importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay subcategoria especificas para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de subcategoria especificas.', $th->getMessage());
        }
    }

    public function importSelectSpecificsubcategory($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $specificsubcategories = $params['dataSpecificsubcategory'];

            $dataSpecificsubcategory = [];
            foreach ($specificsubcategories as $specificsubcategory) {
                $verifySpecificsubcategory = SpecificSubcategory::where('iderp', (int)$specificsubcategory['id'])->active()->first();
                if (!$verifySpecificsubcategory) {
                    $dataSpecificsubcategory[] = [
                        'iderp'             => (int)$specificsubcategory['id'],
                        'name'              => $specificsubcategory['nombre'],
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataSpecificsubcategory)) {
                SpecificSubcategory::insert($dataSpecificsubcategory);

                DB::commit();
                return $this->successResponse('Subcategorias especificas importadas satisfactoriamente.');
            }

            return $this->successResponse('No hay subcategoria especificas para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de subcategoria especificas.', $th->getMessage());
        }
    }

    public function importCustomerOld($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
                    $request['token_erp'] = $token;

            $customers = $this->getCustomerAddressERP($request);
            if (!$customers->original['status']) return $customers;

            $customers = $customers->original['data']['detail'];

            $dataCustomers = [];
            // $dataCustomersLimit = array_filter($customers, function($customer) {
            //     return strpos($customer['nombre_rz'], 'SASVAL') !== false;
            // });
            $dataCustomersLimit = array_slice($customers, 30, 500);
            foreach ($dataCustomersLimit as $customer) {
                $find = strpos($customer['rut'], '-');

                if (!is_null($customer['rut']) && $customer['rut'] != "" && $find !== false) {
                    $proveedor = strtolower($customer['proveedor']);
                    $cliente = strtolower($customer['cliente']);

                    $type = 1;
                    // Validar los casos según las reglas especificadas
                    if ($proveedor == "si" && $cliente == "si") {
                        $type = "3";
                    } elseif (($proveedor == "si") && ($cliente == "no" || $cliente == "" || is_null($cliente))) {
                        $type = "2";
                    } elseif (($cliente == "si") && ($proveedor == "no" || $proveedor == "" || is_null($proveedor))) {
                        $type = "1";
                    }

                    $verifyCustomer = Supplier::where('rut', trim($customer['rut']))->company()->active()->first();
                    if (!$verifyCustomer) {
                        // "rut": "",
                        // "nombre_rz": "",
                        // "nombre_fantasia": "",
                        // "giro": "",
                        // "telefono": "",
                        // "email": "",
                        // "fax": "",
                        // "proveedor": "",
                        // "cliente": "SI",
                        // "prospecto": "",
                        // "observacion": "",
                        // "extranjero": "",
                        // "vendedor": "0",
                        // "web": "",
                        // "estado": 0

                        $dataCustomers[] = [
                            'rut'               => $customer['rut'],
                            'name_rz'           => $customer['nombre_rz'],
                            'fantasy_name'      => $customer['nombre_fantasia'],
                            'web'               => $customer['web'],
                            'giro'              => $customer['giro'],
                            'email'             => $customer['email'],
                            'phone'             => $customer['telefono'],
                            'fax'               => $customer['fax'],
                            'type'              => $type,
                            'legal_nature'      => "39",
                            // 'idwaytopay'        => $customer['nombre'],
                            'idsegment'         => 3, // se debe cambiar
                            // 'idcountry'         => $customer['nombre'],
                            // 'iddepartment'      => $customer['nombre'],
                            // 'idprovince'        => $customer['nombre'],
                            // 'iddistrict'        => $customer['nombre'],
                            // 'address'           => $customer['nombre'],
                            'idcompany'         => $params['idcompany'],
                            'created_at'        => $now,
                            'updated_at'        => $now
                        ];
                    }
                }
            }

            if (!empty($dataCustomers)) {
                Supplier::insert($dataCustomers);

                DB::commit();
                return $this->successResponse('Clientes importados satisfactoriamente.');
            }

            return $this->successResponse('No hay clientes para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de clientes.', $th->getMessage());
        }
    }

    public function importCustomer($params)
    {
        set_time_limit(300);
        DB::beginTransaction();
        try {
            $now = Carbon::now();
            $importedCount = 0;
    
            // Obtener clientes existentes previamente
            $existingClients = Supplier::where('idcompany', $params['idcompany'])->active()->pluck('rut')->toArray();
            
            // Obtener distritos con su jerarquía
            $districts = District::with(['province.department.country'])->get()->keyBy(fn($item) => strtolower($item->name));
           
            DB::connection('mysql_external')->table('clienteproveedor')
                ->select(
                    'RUT_CLIENTEPROVEEDOR AS rut',
                    'RZ_CLIENTEPROVEEDOR AS nombre_rz',
                    'FANTASIA_CLIENTEPROVEEDOR AS nombre_fantasia',
                    'GIRO_CLIENTEPROVEEDOR AS giro',
                    'TELEFONO_CLIENTEPROVEEDOR AS telefono',
                    'EMAIL_CLIENTEPROVEEDOR AS email',
                    'FAX_CLIENTEPROVEEDOR AS fax',
                    'PROVEEDOR_CLIENTEPROVEEDOR AS proveedor',
                    'CLIENTE_CLIENTEPROVEEDOR AS cliente',
                    'PROSPECTO_CLIENTEPROVEEDOR AS prospecto',
                    'OBSERVACION_CLIENTEPROVEEDOR AS observacion',
                    'EXTRANJERO_CLIENTEPROVEEDOR AS extranjero',
                    'ID_VENDEDOR AS vendedor',
                    'PAGINA_WEB AS web'
                )
                ->where('RUT_EMPRESA', $params['rut_company'])
                ->orderBy('RUT_CLIENTEPROVEEDOR')
                ->chunk(500, function ($clients) use ($params, $now, $existingClients, $districts, $importedCount) {
                    $dataClients = [];
                    $clientRuts = [];
                    foreach ($clients as $client) {
                        if (in_array($client->rut, $existingClients)) {
                            continue;
                        }
    
                        $find = strpos($client->rut, '-');
                        if (!is_null($client->rut) && $client->rut != "" && $find !== false ) {
                            $proveedor = strtolower($client->proveedor);
                            $cliente = strtolower($client->cliente);
    
                            $type = 1;
                            if ($proveedor === "si" && $cliente === "si") {
                                $type = "3";
                            } elseif ($proveedor === "si") {
                                $type = "2";
                            } elseif ($cliente === "si") {
                                $type = "1";
                            }
    
                            $dataClients[] = [
                                'rut'               => $client->rut,
                                'name_rz'           => $client->nombre_rz,
                                'fantasy_name'      => $client->nombre_fantasia,
                                'web'               => $client->web,
                                'giro'              => $client->giro,
                                'email'             => $client->email,
                                'phone'             => $client->telefono,
                                'fax'               => $client->fax,
                                'type'              => $type,
                                'legal_nature'      => "39",
                                'idsegment'         => 3,
                                'idcompany'         => $params['idcompany'],
                                'created_at'        => $now,
                                'updated_at'        => $now
                            ];
    
                            $clientRuts[] = $client->rut;
                            $importedCount++;
                        }
                    }
    
                    if (!empty($dataClients)) {
                        Supplier::insert($dataClients);
    
                        // Obtener todas las direcciones de los clientes en un solo bloque
                        $addresses = DB::connection('mysql_external')->table('dir_clienteproveedor')
                            ->select(
                                'RUT_CLIENTEPROVEEDOR AS rut',
                                'ID_DIRECCION AS id',
                                'DIRECCION_DIRECCION AS direccion',
                                'ID_COMUNA AS comuna',
                                'ID_CIUDAD AS ciudad',
                                'EMAIL_DIRECCION AS email',
                                'paises.descripcion AS pais'
                            )
                            ->whereIn('RUT_CLIENTEPROVEEDOR', $clientRuts)
                            ->where('RUT_EMPRESA', $params['rut_company'])
                            ->join('paises', 'paises.codigo', '=', 'dir_clienteproveedor.ID_PAIS')
                            ->orderBy('RUT_CLIENTEPROVEEDOR')
                            ->get();
    
                        $dataAddresses = [];
                        foreach ($addresses as $address) {
                            $district = $districts->get(strtolower($address->comuna));
                            $province = $district ? $district->province : null;
                            $department = $province ? $province->department : null;
                            $country = $department ? $department->country : null;
    
                            $dataAddresses[] = [
                                'address'    => $address->direccion ?? null,
                                'email'      => $address->email ?? null,
                                'idclient'   => Supplier::where('rut', $address->rut)->value('id'),
                                'idcountry'  => $country ? $country->id : null,
                                'idprovince' => $province ? $province->id : null,
                                'iddistrict' => $district ? $district->id : null,
                                'idcompany'  => $params['idcompany']
                            ];
                        }
    
                        if (!empty($dataAddresses)) {
                            ClientAddress::insert($dataAddresses);
                        }
                    }
                });
    
            DB::commit();
            return $this->successResponse("Clientes y direcciones importados satisfactoriamente.");
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('Error durante la importación de clientes.', $th->getMessage());
        }
    }    

    public function importCustomerOld2($params)
    {
        // DB::beginTransaction();
        // try {
            $now = Carbon::now();
    
            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if (!$userErp) return $this->errorResponse('El usuario no cuenta con integración ERP.', 400);
    
            $token = $this->getTokenSoftnet($userErp);
            $request['token_erp'] = $token;
    
            $customers = $this->getCustomerAddressERP($request);
            if (!$customers->original['status']) return $customers;
    
            $customers = $customers->original['data']['detail'];
    
            $dataCustomers = [];
            $dataAddresses = [];
            $dataCustomersLimit = array_slice($customers, 30, 500);
            
            //$dataCustomersLimit = $customers;
            $districts = District::with(['province.department.country'])->get()->keyBy(fn($item) => strtolower($item->name));
    
            foreach ($dataCustomersLimit as $customer) {
                $find = strpos($customer['rut'], '-');
    
                if (!is_null($customer['rut']) && $customer['rut'] != "" && $find !== false) {
                    $proveedor = strtolower($customer['proveedor']);
                    $cliente = strtolower($customer['cliente']);
    
                    $type = 1;
                    if ($proveedor == "SI" && $cliente == "SI") {
                        $type = "3";
                    } elseif (($proveedor == "SI") && ($cliente == "NO" || $cliente == "" || is_null($cliente))) {
                        $type = "2";
                    } elseif (($cliente == "SI") && ($proveedor == "NO" || $proveedor == "" || is_null($proveedor))) {
                        $type = "1";
                    }
    
                    $verifyCustomer = Supplier::where('rut', trim($customer['rut']))->company()->active()->first();
                    if (!$verifyCustomer) {
                        $dataCustomers[] = [
                            'rut'               => $customer['rut'],
                            'name_rz'           => $customer['nombre_rz'],
                            'fantasy_name'      => $customer['nombre_fantasia'],
                            'web'               => $customer['web'],
                            'giro'              => $customer['giro'],
                            'email'             => $customer['email'],
                            'phone'             => $customer['telefono'],
                            'fax'               => $customer['fax'],
                            'type'              => $type,
                            'legal_nature'      => "39",
                            'idsegment'         => 3,
                            'idcompany'         => $params['idcompany'],
                            'created_at'        => $now,
                            'updated_at'        => $now
                        ];
                    }
                }
            }

    
            if (!empty($dataCustomers)) {
                Supplier::insert($dataCustomers);
    
                $insertedSuppliers = Supplier::whereIn('rut', array_column($dataCustomers, 'rut'))->get();
    
                foreach ($insertedSuppliers as $supplier) {
                    foreach ($dataCustomersLimit as $customer) {
                        if ($customer['rut'] === $supplier->rut && isset($customer['direccion']) && is_array($customer['direccion'])) {
                            foreach ($customer['direccion'] as $direccion) {
                                $district = $districts->get(strtolower($direccion['comuna']));
                                $province = $district ? $district->province : null;
                                $department = $province ? $province->department : null;
                                $country = $department ? $department->country : null;
    
                                $dataAddresses[] = [
                                    'address'    => $direccion['direccion'] ?? null,
                                    'email'      => $direccion['email'] ?? null,
                                    'idclient'   => $supplier->id,
                                    'idcountry'  => $country ? $country->id : null,
                                    'idprovince' => $province ? $province->id : null,
                                    'iddistrict' => $district ? $district->id : null,
                                    'idcompany'  => $params['idcompany']
                                ];
                            }
                        }
                    }
                }
    
                // Inserción masiva de direcciones
                if (!empty($dataAddresses)) {
                    ClientAddress::insert($dataAddresses);
                }
                return $this->successResponse('Clientes y direcciones importados satisfactoriamente.');

            }
    
            // DB::commit();
            return $this->successResponse('No hay clientes nuevos.');
        // } catch (\Throwable $th) {
        //     DB::rollBack();
        //     return $this->externalError('durante la importación de clientes.', $th->getMessage());
        // }
    }
    

    public function importSelectCustomer($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $customers = $params['dataCustomer'];

            $dataCustomers = [];
            foreach ($customers as $customer) {
                $find = strpos($customer['rut'], '-');

                if (!is_null($customer['rut']) || $customer['rut'] != "" || $find !== false) {
                    $proveedor = strtolower($customer['proveedor']);
                    $cliente = strtolower($customer['cliente']);

                    $type = 1;
                    // Validar los casos según las reglas especificadas
                    if ($proveedor == "si" && $cliente == "si") {
                        $type = "3";
                    } elseif (($proveedor == "si") && ($cliente == "no" || $cliente == "" || is_null($cliente))) {
                        $type = "2";
                    } elseif (($cliente == "si") && ($proveedor == "no" || $proveedor == "" || is_null($proveedor))) {
                        $type = "1";
                    }

                    $verifyCustomer = Supplier::where('rut', trim($customer['rut']))->company()->active()->first();
                    if (!$verifyCustomer) {
                        $dataCustomers[] = [
                            'rut'               => trim($customer['rut']),
                            'name_rz'           => $customer['nombre_rz'],
                            'fantasy_name'      => $customer['nombre_fantasia'],
                            'web'               => $customer['web'],
                            'giro'              => $customer['giro'],
                            'email'             => $customer['email'],
                            'phone'             => $customer['telefono'],
                            'fax'               => $customer['fax'],
                            'type'              => $type,
                            'legal_nature'      => "39",
                            // 'idwaytopay'        => $customer['nombre'],
                            'idsegment'         => 3, // se debe cambiar
                            // 'idcountry'         => $customer['nombre'],
                            // 'iddepartment'      => $customer['nombre'],
                            // 'idprovince'        => $customer['nombre'],
                            // 'iddistrict'        => $customer['nombre'],
                            // 'address'           => $customer['nombre'],
                            'idcompany'         => $params['idcompany'],
                            'created_at'        => $now,
                            'updated_at'        => $now
                        ];
                    }
                }
            }

            if (!empty($dataCustomers)) {
                Supplier::insert($dataCustomers);

                DB::commit();
                return $this->successResponse('Clientes importados satisfactoriamente.');
            }

            return $this->successResponse('No hay clientes para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de clientes.', $th->getMessage());
        }
    }

    public function importSupplier($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
                    $request['token_erp'] = $token;

            $suppliers = $this->getSupplierERP($request);
            if (!$suppliers->original['status']) return $suppliers;

            $suppliers = $suppliers->original['data']['detail'];

            $dataSupplier = [];
            $dataSuppliersLimit = array_slice($suppliers, 0, 30);
            foreach ($dataSuppliersLimit as $supplier) {
                $find = strpos($supplier['rut'], '-');

                if (!is_null($supplier['rut']) && $supplier['rut'] != "" && $find !== false) {
                    $proveedor = strtolower($supplier['proveedor']);
                    $cliente = strtolower($supplier['cliente']);

                    $type = 1;
                    // Validar los casos según las reglas especificadas
                    if ($proveedor == "si" && $cliente == "si") {
                        $type = "3";
                    } elseif (($proveedor == "si") && ($cliente == "no" || $cliente == "" || is_null($cliente))) {
                        $type = "2";
                    } elseif (($cliente == "si") && ($proveedor == "no" || $proveedor == "" || is_null($proveedor))) {
                        $type = "1";
                    }

                    $verifySupplier = Supplier::where('rut', trim($supplier['rut']))->company()->active()->first();
                    if (!$verifySupplier) {
                        $dataSupplier[] = [
                            'rut'               => $supplier['rut'],
                            'name_rz'           => $supplier['nombre_rz'],
                            'fantasy_name'      => $supplier['nombre_fantasia'],
                            'web'               => $supplier['web'],
                            'giro'              => $supplier['giro'],
                            'email'             => $supplier['email'],
                            'phone'             => $supplier['telefono'],
                            'fax'               => $supplier['fax'],
                            'type'              => $type,
                            'legal_nature'      => "39",
                            // 'idwaytopay'        => $supplier['nombre'],
                            'idsegment'         => 3, // se debe cambiar
                            // 'idcountry'         => $supplier['nombre'],
                            // 'iddepartment'      => $supplier['nombre'],
                            // 'idprovince'        => $supplier['nombre'],
                            // 'iddistrict'        => $supplier['nombre'],
                            // 'address'           => $supplier['nombre'],
                            'idcompany'         => $params['idcompany'],
                            'created_at'        => $now,
                            'updated_at'        => $now
                        ];
                    }
                }
            }

            if (!empty($dataSupplier)) {
                Supplier::insert($dataSupplier);

                DB::commit();
                return $this->successResponse('Proveedores importados satisfactoriamente.');
            }

            return $this->successResponse('No hay proveedores para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de proveedores.', $th->getMessage());
        }
    }

    public function importSelectSupplier($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $suppliers = $params['dataSupplier'];

            $dataSupplier = [];
            foreach ($suppliers as $supplier) {
                $find = strpos($supplier['rut'], '-');

                if (!is_null($supplier['rut']) && $supplier['rut'] != "" && $find !== false) {
                    $proveedor = strtolower($supplier['proveedor']);
                    $cliente = strtolower($supplier['cliente']);

                    $type = 1;
                    // Validar los casos según las reglas especificadas
                    if ($proveedor == "si" && $cliente == "si") {
                        $type = "3";
                    } elseif (($proveedor == "si") && ($cliente == "no" || $cliente == "" || is_null($cliente))) {
                        $type = "2";
                    } elseif (($cliente == "si") && ($proveedor == "no" || $proveedor == "" || is_null($proveedor))) {
                        $type = "1";
                    }

                    $verifySupplier = Supplier::where('rut', trim($supplier['rut']))->company()->active()->first();
                    if (!$verifySupplier) {
                        $dataSupplier[] = [
                            'rut'               => $supplier['rut'],
                            'name_rz'           => $supplier['nombre_rz'],
                            'fantasy_name'      => $supplier['nombre_fantasia'],
                            'web'               => $supplier['web'],
                            'giro'              => $supplier['giro'],
                            'email'             => $supplier['email'],
                            'phone'             => $supplier['telefono'],
                            'fax'               => $supplier['fax'],
                            'type'              => $type,
                            'legal_nature'      => "39",
                            // 'idwaytopay'        => $supplier['nombre'],
                            'idsegment'         => 3, // se debe cambiar
                            // 'idcountry'         => $supplier['nombre'],
                            // 'iddepartment'      => $supplier['nombre'],
                            // 'idprovince'        => $supplier['nombre'],
                            // 'iddistrict'        => $supplier['nombre'],
                            // 'address'           => $supplier['nombre'],
                            'idcompany'         => $params['idcompany'],
                            'created_at'        => $now,
                            'updated_at'        => $now
                        ];
                    }
                }
            }

            if (!empty($dataCustomers)) {
                Supplier::insert($dataCustomers);

                DB::commit();
                return $this->successResponse('Proveedores importados satisfactoriamente.');
            }

            return $this->successResponse('No hay proveedores para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de proveedores.', $th->getMessage());
        }
    }

    public function importBranchOffice($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);
                    $request['token_erp'] = $token;

            $branchOffices = $this->getBranchOfficeERP($request);
            if (!$branchOffices->original['status']) return $branchOffices;

            $branchOffices = $branchOffices->original['data']['detail'];

            $dataBranchOffices = [];
            foreach ($branchOffices as $branchOffice) {
                $verifyBranchOffices = BranchOffice::where('iderp', $branchOffice['id'])->company()->active()->first();
                if (!$verifyBranchOffices) {
                    $comune = District::where('name', $branchOffice['comuna'])->active()->first();
                    if ($comune) {
                        $comune = DistrictResource::make($comune->load('province'));
                        $comune = json_decode(json_encode($comune));
                    }

                    $dataBranchOffices[] = [
                        'iderp'             => $branchOffice['id'],
                        'code'              => 'ERP-' . $branchOffice['id'],
                        'name'              => $branchOffice['nombre'],
                        'address'           => $branchOffice['direccion'],
                        'idofficetype'      => 1,

                        'iddistrict'        => $comune->id ?? null,
                        'idprovince'        => $comune->idprovince ?? null,
                        'iddepartment'      => $comune->iddepartment ?? null,
                        'idcountry'         => $comune->idcountry ?? null,
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataBranchOffices)) {
                BranchOffice::insert($dataBranchOffices);

                DB::commit();
                return $this->successResponse('Sucursales importados satisfactoriamente.');
            }

            return $this->successResponse('No hay sucursales para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de sucursales.', $th->getMessage());
        }
    }

    public function importSelectBranchOffice($params)
    {
        DB::beginTransaction();
        try {
            $now = Carbon::now();

            $branchOffices = $params['dataBranchOffice'];

            $dataBranchOffices = [];
            foreach ($branchOffices as $branchOffice) {
                $verifyBranchOffices = BranchOffice::where('iderp', $branchOffice['id'])->company()->active()->first();
                if (!$verifyBranchOffices) {
                    $comune = District::where('name', $branchOffice['comuna'])->active()->first();
                    if ($comune) {
                        $comune = DistrictResource::make($comune->load('province'));
                        $comune = json_decode(json_encode($comune));
                    }

                    $dataBranchOffices[] = [
                        'iderp'             => $branchOffice['id'],
                        'code'              => 'ERP-' . $branchOffice['id'],
                        'name'              => $branchOffice['nombre'],
                        'address'           => $branchOffice['direccion'],
                        'idofficetype'      => 1,

                        'iddistrict'        => $comune->id ?? null,
                        'idprovince'        => $comune->idprovince ?? null,
                        'iddepartment'      => $comune->iddepartment ?? null,
                        'idcountry'         => $comune->idcountry ?? null,
                        'idcompany'         => $params['idcompany'],
                        'created_at'        => $now,
                        'updated_at'        => $now
                    ];
                }
            }

            if (!empty($dataBranchOffices)) {
                BranchOffice::insert($dataBranchOffices);

                DB::commit();
                return $this->successResponse('Sucursales importados satisfactoriamente.');
            }

            return $this->successResponse('No hay sucursales para importar.');
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la importación de sucursales.', $th->getMessage());
        }
    }

    public function issueTicket($id, $params)
    {
        try {
            DB::beginTransaction();

            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);

            # Validar ticket
            $ticket = Ticket::activeForID($id)->first();
            if (!$ticket) return $this->errorResponse('El ticket seleccionado es inválido.', 400);
            if ($ticket->status_invoice == 1) return $this->errorResponse('El ticket ya ha sido emitido.', 400);

            $detailTicket = DetailTicket::where('idticket', $ticket->id)->active()->get();

            # Inicio de petición
            $client = new Client();
            $headers = ['token' => $token];

            # Valores generales
            $date = Date::now()->format('j/n/Y, H:i:s');

            # Buscamos la direccion
            $clientAddress = ClientAddress::where('id', $ticket->idclient_address)->first();

            # Variables para Encabezado
            $receiver = $ticket['customer']['rut']; 
            $netAmount = $ticket->sub_total;
            $exemptAmount = $ticket->mnt_excent;
            $discountHeader = $ticket->total_discount;
            $discountType = "1";
            $folio = 0; # TEST
            $businessArea = $ticket->bussines_area; 
            $seller = $ticket->seller_rut ?? '15134573-5'; # TEST - DE LA SESIÓN
            $observation = $ticket->observation;
            $address = $clientAddress ? $clientAddress->id :"SANTIAGO";

            #bodega 
            $store = Store::where('idcompany', $ticket->idcompany)->where('status', 1)->first();
            if(!$store) return $this->errorResponse('La bodega no valida', 400);
            # Variables para Detalle
            $detailJson = [];
            $reduceProducts = [];
            foreach ($detailTicket as $detail) {

                $detailJson[] = [
                    "Codigo"    => $detail['productOrigin']['code_erp'], # code_erp o code_system
                    "Cantidad"  => $detail->amount,
                    "Descuento" => $detail->discounttype == 1? $detail->discount : ($detail->amount * $detail->price)  * ($detail->discount /100),
                    "Precio"    => $detail->price,
                    "TipoDescuento" =>  "1",
                    "Afecto"    => true,
                    "Bodega"    => $store->iderp, # TEST - Se obtiene la bodega cuando se importa, si es creado de 0 no tiene warehouse, mandar uno por defecto?
                ];

                $reduceProducts[] = [
                    'idproduct' => $detail['productOrigin']['id'],
                    'amount'    => $detail->amount
                ];
            }

            #Forma de Pago
            $wayPay = WayToPay::activeForID($ticket->idway_to_pay)->first();
            $condicionPago = $wayPay->iderp;
            # Variables para Adicional
            
            $one = $wayPay->name; # TEST EFECTIVO
            $two = $ticket->seller_rut ?? '15134573-5'; # TEST - DE LA SESIÓN

            
            # Variable para Cliente
            $dataClient = 'cliente';

            switch ($ticket->idtype_document) {
                case '1':
                    # Boleta electrónica

                    $body = '[
                        {
                            "Encabezado": {
                                "Receptor": "' . $receiver . '",
                                "MontoNeto": ' . $netAmount . ',
                                "Descuento": ' . $discountHeader . ',
                                "TipoDocumento": "39",
                                "TipoDescuento": ' . $discountType . ',
                                "AreaNegocio": ' . $businessArea . ',
                                "Observacion": "' . $observation . '",
                                "Direccion": "' . $address . '"
                            },
                            "Detalle": ' . json_encode($detailJson) . ',
                            "cliente": "' . $dataClient . '",
                            "Adicional": {
                                "Uno": ' . $condicionPago . ',
                                "Dos": "' . $two . '"
                            },
                            "fecha": "' . $date . '"
                        }
                    ]'; 

                    $request = new Request('POST', 'api.softnet.cl/boleta', $headers, $body);
                    $endpoint = 'api.softnet.cl/boleta';

                    break;
                case '2':
                    # Factura
                    $body = '[
                        {
                            "Encabezado": {
                                "Receptor": "' . $receiver . '",
                                "MontoNeto": ' . $netAmount . ',
                                "MontoExento": "' . $exemptAmount . '",
                                "Descuento": "' . $discountHeader . '",
                                "TipoDocumento": "33",
                                "Folio": ' . $folio . ',
                                "AreaNegocio": "' . $businessArea . '",
                                "Vendedor": "' . $seller . '",
                                "Observacion": "' . $observation . '",
                                "Direccion": "' . $address . '"
                            },
                            "Detalle": ' . json_encode($detailJson) . ',
                            "Adicional": {
                                "Uno": ' . $condicionPago . ',
                                "Dos": "' . $two . '"
                            }
                        }
                    ]';

                    $request = new Request('POST', 'api.softnet.cl/factura', $headers, $body);
                    $endpoint = 'api.softnet.cl/factura';

                    break;

                case '3':
                    # Nota de crédito

                    #Validación interna
                    $validation = Validator::make($params, [
                        'number'    => 'required|integer|min:1',
                        'type'      => 'required|in:33,34,39',
                        'folio'     => 'required|integer|min:1',
                        'date'      => 'required|date',
                        'code_nc'   => 'required|integer|min:1',
                        'reason'    => 'required|string',
                    ]);
                    if ($validation->fails()) return $this->errorResponse('Formato inválido.', 400, $validation->errors());

                    /* Condicion de pago - TEST
                    Direccion debe ser entero*/

                    $body = '[
                        {
                          "Encabezado": {
                            "Receptor": "' . $receiver . '",
                            "Folio": ' . $folio . ',
                            "MontoNeto": ' . $netAmount . ',
                            "MontoExento": "' . $exemptAmount . '",
                            "Descuento": "' . $discountHeader . '",
                            "TipoDocumento": 61,
                            "AreaNegocio": "' . $businessArea . '",
                            "Vendedor": "' . $seller . '",
                            "CondicionPago": "2400",
                            "Observacion": "' . $observation . '"
                          },
                          "Detalle": ' . json_encode($detailJson) . ',
                          "Referencia": {
                            "Numero":"' . $params['number'] . '",
                            "Tipo":"' . $params['type'] . '",
                            "Folio":"' . $params['folio'] . '",
                            "Fecha":"' . $params['date'] . '",
                            "CodigoNC":"' . $params['code_nc'] . '",
                            "Razon":"' . $params['reason'] . '"
                          },
                          "Adicional": {
                            "Uno": "' . $one . '",
                            "Dos": "' . $two . '"
                          }
                        }
                      ]';

                    $request = new Request('POST', 'api.softnet.cl/notaCredito', $headers, $body);
                    $endpoint = 'api.softnet.cl/notaCredito';

                    break;

                case '4':
                    # Nota de débito
                    return $this->errorResponse('La nota de débito no esta disponible.', 400);

                    break;

                case '5':
                    # Cotización

                    /* Condicion de pago - TEST
                    Direccion debe ser entero*/

                    $body = '[
                            {
                                "Encabezado": {
                                    "Receptor": "' . $receiver . '",
                                    "MontoNeto": ' . $netAmount . ',
                                    "MontoExento": "' . $exemptAmount . '",
                                    "Descuento": "' . $discountHeader . '",
                                    "TipoDocumento": "777",
                                    "Folio": ' . $folio . ',
                                    "AreaNegocio": "' . $businessArea . '",
                                    "CondicionPago": "' . $condicionPago . '",
                                    // "Vendedor": "' . $seller . '",
                                    "Vigencia":"-",
                                    "Observacion": "' . $observation . '",
                                    "Direccion": "123"
                                },
                                "Detalle": ' . json_encode($detailJson) . ',
                                "Adicional": {
                                    "Uno": "' . $one . '",
                                    "Dos": "' . $two . '"
                                }
                            }
                        ]';

                    $request = new Request('POST', 'api.softnet.cl/cotizacion', $headers, $body);
                    $endpoint = 'api.softnet.cl/cotizacion';

                    break;

                case '6':
                    # Nota de venta

                    /* Condicion de pago - TEST
                    Direccion debe ser entero*/
                    $body = '[
                        {
                            "Encabezado": {
                                "Receptor": "' . $receiver . '",
                                "MontoNeto": ' . $netAmount . ',
                                "MontoExento": "' . $exemptAmount . '",
                                "Descuento": "' . $discountHeader . '",
                                "TipoDocumento": "780",
                                "Folio": ' . $folio . ',
                                "AreaNegocio": "' . $businessArea . '",
                                "CondicionPago": "' . $condicionPago . '",
                                "Vendedor": "' . $seller . '",
                                "Observacion": "' . $observation . '",
                                "Direccion": "123"
                            },
                            "Detalle": ' . json_encode($detailJson) . ',
                            "cliente": "' . $dataClient . '",
                            "Adicional": {
                                "Uno": "' . $one . '",
                                "Dos": "' . $two . '"
                            }
                        }
                    ]';

                    $request = new Request('POST', 'api.softnet.cl/notaVenta', $headers, $body);
                    $endpoint = 'api.softnet.cl/notaVenta';

                    break;
                default:
                    return $this->errorResponse('Emisión inválida.', 400);
                    break;
            }

            # Esperar petición
            $res = $client->sendAsync($request)->wait();
            $res = $res->getBody();
            $resJson = json_decode($res)[0] ?? false;
            //dd($request);

            // Verificar respuesta de API softnet
            if (!$resJson) return $this->errorResponse('Inténtelo nuevamente en unos momentos.', 400);

            # Actualizar ticket
            $ticket->update([
                'status_invoice'    => 1,
                'url_invoice'       => $resJson->pdf ?? null,
                'folio'             => $resJson->folio ?? null
            ]);

            # Generar registro log del uso de API
            LogApiERP::create([
                'idticket'  => $id,
                'endpoint'  => $endpoint,
                'metodo'    => 'POST',
                'entrada'   => $body,
                'respuesta' => $res ?? ''
            ]);

            # Reducir stock por emisión del ticket en lista de precios o al stock del producto
            foreach ($reduceProducts as $product) {
                $detailPriceListProduct = DetailPriceListProduct::where('idproduct', $product['idproduct'])
                    ->company()->active()->first();

                if ($detailPriceListProduct) {
                    $detailPriceListProduct->decrement('stock', $product['amount']);
                } else {
                    # Busco el stock
                    $stockProduct = StockProduct::where('idproduct', $product['idproduct'])
                        ->company()->active()->first();
                    // $newQuantity = $stockProduct->stock_system - $product['amount'];
                    // $stockProduct->stock_system = $newQuantity;
                    // $stockProduct->update();
                    // $stockProduct->decrement('stock_system', $product['amount']);
                }
            }
            DB::commit();
            return $this->successResponse('Emisión de ticket completado.', $resJson);
        } catch (\Throwable $th) {
            throw $th;
            DB::rollBack();
            return $this->externalError('durante la emisión de un ticket.', $th->getMessage());
        }
    }

    public function checkFolio($folio)
    {
        try {
            $userErp = UserERP::where('iduser', Auth::user()->id)->first();
            if(!$userErp) return $this->errorResponse('El usuario no cuenta con integracion ERP.', 400);

            $token = $this->getTokenSoftnet($userErp);

            # Inicio de petición
            $client = new Client();
            $headers = ['token' => $token];
            $request = new Request('GET', 'api.softnet.cl/cargarcaf/' . $folio, $headers);

            # Esperar petición
            $res = $client->sendAsync($request)->wait();
            $res = $res->getBody();
            $resJson = json_decode($res);
            return $this->successResponse($resJson);
        } catch (\Throwable $th) {
            return $this->errorResponse('Ocurrió un error al verificar el folio.', 400);
        }
    }
}
