<?php

namespace App\Services;

use App\Http\Resources\PriceListProductResource;
use App\Http\Resources\ProductResource;
use App\Http\Resources\ShoppingCartResource;
use App\Models\DetailPriceListProduct;
use App\Models\DetailShoppingCart;
use App\Models\PriceList;
use App\Models\Product;
use App\Models\ShoppingCart;
use App\Models\StockProduct;
use App\Models\SettingBilling;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class ShoppingCartService
 * @package App\Services
 */
class ShoppingCartService
{
    use HasResponse;

    public $igv;

    public function __construct()
    {
        $this->igv = 0.19;
    }

    public function list($withPagination)
    {
        $shoppingCart = ShoppingCart::shoppingCartFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc');

        $shoppingCart = !empty($withPagination)
            ? $shoppingCart->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $shoppingCart->get();

        $shoppingCart = ShoppingCartResource::collection($shoppingCart->load('user', 'customer', 'company'));

        return $this->successResponse('Lectura exitosa.', $shoppingCart);
    }

    public function listCustomer($id, $params)
    {
        $shoppingCart = ShoppingCart::shoppingCartCustomer($id, $params['iduser_created'])->active()->first();

        if ($shoppingCart) {
            $shoppingCart = ShoppingCartResource::make($shoppingCart->load('user', 'customer', 'company'));
        } else {
            $shoppingCart = [];
        }

        return $this->successResponse('Lectura exitosa.', $shoppingCart);
    }

    public function changeGlobalDiscount($id, $params)
    {
        DB::beginTransaction();
        try {
            $shoppingCart = ShoppingCart::activeForID($id)->company()->first();
            if (!$shoppingCart) return $this->errorResponse('Carrito de compras seleccionado no encontrado', 400);

            $productsShoppingCart = DetailShoppingCart::where('idshopping_cart', $shoppingCart->id)->active();
            $idsDetail = $productsShoppingCart->pluck('id');
            if (count($idsDetail) == 0) return $this->errorResponse('Debes de tener por lo menos un producto en el carrito', 400);

            if ($params['global_discount'] > 0) {

                $settingBilling = SettingBilling::where('idcompany', $params['idcompany'])->where('documenttype', $shoppingCart->documenttype)->first();
                if(!$settingBilling) return $this->errorResponse('La empresa no cuenta con configuracion por empresa', 400);

                #calculos con netos
            
                $totalProductDiscounts = $productsShoppingCart->sum('discount');
                $totalProductPrice = $productsShoppingCart->sum('gross_subtotal');

                $shoppingCart->global_discount = $params['global_discount'];
                $shoppingCart->discounttype = $params['discounttype'];
                $grossTotalProducts = $totalProductPrice;

                if ( $settingBilling->calculationtype  === "1") {

                    $sub_total = $grossTotalProducts;
                    $shoppingCart->sub_total = $sub_total;
                
                }

                if ( $settingBilling->calculationtype  === "2") {

                    $sub_total = ($grossTotalProducts) / (1 + $this->igv);
                    $shoppingCart->sub_total = $sub_total;
                }

                $discountGlobalMount = $params['discounttype'] === 2 ? ($sub_total * ($params['global_discount']/100)) :  $params['global_discount'];
    
                $net_total = $sub_total - $discountGlobalMount;
                $shoppingCart->net_total =  $net_total;
    
                $shoppingCart->IGV = $net_total * $this->igv;
                $shoppingCart->gross_total = $net_total + ($net_total * $this->igv);
                
                $shoppingCart->total = $grossTotalProducts - $params['global_discount'];
            
                $shoppingCart->total_discount = $params['global_discount'] + $totalProductDiscounts;

                $shoppingCart->total = $grossTotalProducts;
                $shoppingCart->save();

            }

            DB::commit();
            return $this->successResponse('Descuento global editado satisfactoriamente.', $shoppingCart);

        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización del descuento global.', $th->getMessage());
        }
    }

    public function changeGlobalDiscountOld($id, $params)
    {
        DB::beginTransaction();
        try {
            $shoppingCart = ShoppingCart::activeForID($id)->company()->first();
            if (!$shoppingCart) return $this->errorResponse('Carrito de compras seleccionado no encontrado', 400);

            $productsShoppingCart = DetailShoppingCart::where('idshopping_cart', $shoppingCart->id)->active();

            // $idsPriceList = DetailPriceListProduct::whereIn('id', $productsShoppingCart)->active()->groupBy('idpricelist')->pluck('idpricelist');

            $idsDetail = $productsShoppingCart->pluck('id');

            if (count($idsDetail) == 0) return $this->errorResponse('Debes de tener por lo menos un producto en el carrito', 400);

            // if (count($idsPriceList) > 0) {
            //     $max_desc_global = PriceList::whereIn('id', $idsPriceList)->active()->max('max_desc_global');
            //     if ($params['global_discount'] > $max_desc_global) return $this->errorResponse('El descuento global sobrepasada el maximo impuesto', 400);
            // }

            if ($params['global_discount'] > 0) {
                $grossTotal = $shoppingCart->gross_total;
                $netTotal = $shoppingCart->gross_total / (1 + $this->igv);
                $IGV = $netTotal * $this->igv;

                $updateShoppingCart = ShoppingCart::find($id);
                $updateShoppingCart->global_discount = $params['global_discount'];
                $updateShoppingCart->gross_total = $grossTotal;
                $updateShoppingCart->total_discount = $params['global_discount'];
                $updateShoppingCart->net_total = $netTotal;
                $updateShoppingCart->IGV = $IGV;
                $updateShoppingCart->total = $grossTotal - $params['global_discount'];
                $updateShoppingCart->save();

                // $detailShoppingCart = DetailShoppingCart::where('idshopping_cart', $shoppingCart->id)->active()->get();
                // foreach ($detailShoppingCart as $detail) {
                //     $updateDetailSC = DetailShoppingCart::find($detail->id);
                //     $updateDetailSC->offer_price = $detail->price;
                //     $updateDetailSC->discount = 0;
                //     $updateDetailSC->net_subtotal = $detail->gross_subtotal;
                //     $updateDetailSC->save();
                // }

                DB::commit();
                return $this->successResponse('Descuento global editado satisfactoriamente.', $updateShoppingCart);
            }
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización del descuento global.', $th->getMessage());
        }
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {

            $customer = null;
            if (isset($params['idcustomer'])) $customer = $params['idcustomer'];
            $quantity = $params['quantity'];

            # Verificar si existe el carrito para el cliente
            $shoppingCart = ShoppingCart::shoppingCartCustomer($params['iduser_created'], $customer)->where('status_ticket', 0)->company()->active()->first();
            if (!$shoppingCart) {
                $shoppingCart = ShoppingCart::create([
                    'iduser_created'    => $params['iduser_created'],
                    'idcustomer'        => $customer,
                    'documenttype'      => 1,
                    'discounttype'      => 1,
                    'idcompany'         => $params['idcompany']
                ]);
                $shoppingCart->fresh();
            }

            $settingBilling = SettingBilling::where('idcompany', $params['idcompany'])->where('documenttype', $shoppingCart->documenttype)->first();
            if(!$settingBilling) return $this->errorResponse('La empresa no cuenta con configuracion por empresa', 400);


            #verificamos si el producto ya fue agregado
            $verifyProduct = DetailShoppingCart::where('idproduct', $params['idproduct'])->where('idshopping_cart', $shoppingCart->id)->active()->first();
            if($verifyProduct) {
                $quantity = $verifyProduct->quantity + $params['quantity'];

                $verifyProduct->quantity = $quantity;
                $grossSubtotal = ($verifyProduct->price * $quantity);
                $grossSubtotal = $verifyProduct->discounttype  === 2 ? $grossSubtotal * ((100 - $verifyProduct->discount) / 100) : $grossSubtotal - $verifyProduct->discount;
                $verifyProduct->gross_subtotal =  $grossSubtotal;
                $verifyProduct->net_subtotal = $grossSubtotal;
                $verifyProduct->save();
    
            }else{

                $product = Product::activeForID($params['idproduct'])->first();

                $detailShoppingCart = DetailShoppingCart::create([
                    'idshopping_cart'       => $shoppingCart->id,
                    'idpricelist_product'   => null,
                    'idproduct'             => $params['idproduct'],
                    'quantity'              => $params['quantity'],
                    'price'                 => $product->price,
                    'discount'              => 0,
                    'discounttype'          => 1,
                    'gross_subtotal'        => $params['quantity'] * $product->price,
                    'net_subtotal'          => $params['quantity'] * $product->price
                ]);

                $detailShoppingCart->save();
            }

            #actualizamos el carrito
            $productsShoppingCart = DetailShoppingCart::where('idshopping_cart', $shoppingCart->id)->active();
            $totalProductDiscounts = $productsShoppingCart->sum('discount');
            $grossTotalProducts = $productsShoppingCart->sum('gross_subtotal');

            #calculo con netos
            if ( $settingBilling->calculationtype  === "1") {
                
                $sub_total = $grossTotalProducts;
                $shoppingCart->sub_total = $sub_total;
            
            }

            #calculo con brutos
            if ( $settingBilling->calculationtype  === "2") {

                $sub_total = ($grossTotalProducts) / (1 + $this->igv);
                $shoppingCart->sub_total = $sub_total;
            }

            $discountGlobalMount = $shoppingCart->discounttype === 2 ? ($sub_total * ($shoppingCart->global_discount/100)) :  $shoppingCart->global_discount;
            $net_total = $sub_total - $discountGlobalMount;
            $shoppingCart->net_total =  $net_total;

            $shoppingCart->IGV = $net_total * $this->igv;
            $shoppingCart->gross_total = $net_total + ($net_total * $this->igv);
            
            $shoppingCart->total = $grossTotalProducts - $shoppingCart->global_discount;
            
            $shoppingCart->save();


            DB::commit();
            return $this->successResponse('Producto agregado al carrito satisfactoriamente.', $shoppingCart);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante el agregado del producto al carrito.', $th->getMessage());
        }
    }

    public function registerOld($params)
    {
        DB::beginTransaction();
        try {
            $customer = null;
            if (isset($params['idcustomer'])) $customer = $params['idcustomer'];
            $quantity = $params['quantity'];

            # Verificar si existe el carrito para el cliente
            $shoppingCart = ShoppingCart::shoppingCartCustomer($params['iduser_created'], $customer)->where('status_ticket', 0)->company()->active()->first();
            if (!$shoppingCart) {
                $shoppingCart = ShoppingCart::create([
                    'iduser_created'    => $params['iduser_created'],
                    'idcustomer'        => $customer,
                    'idcompany'         => $params['idcompany']
                ]);
                $shoppingCart->fresh();
            }

            $productList = DetailPriceListProduct::where('idproduct', $params['idproduct'])->company()->first();
            $product = Product::activeForID($params['idproduct'])->first();

            if ($productList) {
                $productList = PriceListProductResource::make($productList->load('product', 'pricelist'));
                $productList = json_decode(json_encode($productList));
                $stock = $productList->stock;
                $idregister = $productList->id;
                $type = 1;
                $price = $productList->price;
                $discounttype_pricelist =  $productList->discounttype_pricelist;
            }else {
                $productStock = StockProduct::where('idproduct', $product->id)->company()->active()->first();
                $stock = $productStock->stock_system ?? 0;
                $idregister = $product->id;
                $type = 2;
                $price = $product->price;
                $discounttype_pricelist = 3; # Sin descuento
            }

            // if ($stock < $params['quantity']) return $this->errorResponse('No puedes exceder la cantidad de productos que hay en stock.', 400);

            # Verificamos si el producto ya fue agregado al carrito
            $verifyProduct = DetailShoppingCart::productShoppingCart($shoppingCart->id, $idregister, $type)->active()->first();
            $oldDiscount = 0;
            $grossSubtotal = 0;
            $netSubtotal = 0;
            if ($verifyProduct) {
                $quantity = $verifyProduct->quantity + $params['quantity'];

                $detailShoppingCart = DetailShoppingCart::updateOrCreate(['id' => $verifyProduct->id], [
                    'quantity'              => $quantity,
                    'gross_subtotal'        => $quantity * $price,
                    'net_subtotal'          => $this->calculateGrossSubtotal($quantity, $price, $verifyProduct->discount, $discounttype_pricelist)
                ]);

                $oldDiscount = $verifyProduct->gross_subtotal - $verifyProduct->net_subtotal;
                // $discount = $verifyProduct->discount;
                $grossSubtotal = $verifyProduct->gross_subtotal;
                $netSubtotal = $verifyProduct->net_subtotal;
            } else {
                # Creamos el detalle del carrito
                $detailShoppingCart = DetailShoppingCart::create([
                    'idshopping_cart'       => $shoppingCart->id,
                    'idpricelist_product'   => $productList->id ?? null,
                    'idproduct'             => $product->id,
                    'quantity'              => $params['quantity'],
                    'price'                 => $price,
                    'discount'              => 0,
                    'discounttype'          => $discounttype_pricelist,
                    'gross_subtotal'        => $params['quantity'] * $price,
                    'net_subtotal'          => $this->calculateGrossSubtotal($params['quantity'], $price, 0, $discounttype_pricelist)
                ]);
            }
            $detailShoppingCart->fresh();

            $newDiscount = $detailShoppingCart->net_subtotal - $detailShoppingCart->gross_subtotal;

            #Actualizo el total en el carrito
            $data['gross_total'] = ($shoppingCart->gross_total - $grossSubtotal) + $detailShoppingCart->gross_subtotal;
            $data['net_total'] = $data['gross_total'] / (1 + $this->igv);
            $data['IGV'] = $data['net_total'] * $this->igv;

            # Valido si tiene descuento global
            if ($shoppingCart->global_discount > 0) {
                $data['total'] = $data['gross_total'] - $shoppingCart->global_discount;
            } else {
                $data['total_discount'] = ($shoppingCart->total_discount - $oldDiscount) + $newDiscount;
                $data['total'] = $data['gross_total'] - $data['total_discount'];
            }

            $updateShoppingCart = ShoppingCart::updateOrCreate(['id' => $shoppingCart->id], $data);
            $updateShoppingCart->fresh();

            DB::commit();
            return $this->successResponse('Producto agregado al carrito satisfactoriamente.', $updateShoppingCart);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante el agregado del producto al carrito.', $th->getMessage());
        }
    }

    public function updateProduct($id, $params)
    {
        DB::beginTransaction();
        try {
           
            $detailShoppingCart = DetailShoppingCart::activeForID($id)->first();
            $quantity = $params['quantity'] ?? $detailShoppingCart->quantity;
            $discount = $params['discount'] ?? $detailShoppingCart->discount;
            $typeDiscount = $params['discountType'] ?? (int)$detailShoppingCart->discounttype;
            
            $detailShoppingCart->quantity = $quantity;
            $detailShoppingCart->discount = $discount;
            $detailShoppingCart->discounttype =  $typeDiscount;
            $subtotalItem = $detailShoppingCart->price * $quantity;
            $grossSubtotal =  $typeDiscount === 2 ? ($subtotalItem * ( 1 - ($discount/100) )) :  $subtotalItem - $discount;
            $detailShoppingCart->gross_subtotal =  $grossSubtotal;
            $detailShoppingCart->net_subtotal = $grossSubtotal;

            $detailShoppingCart->save();

            $shoppingCart = ShoppingCart::find($detailShoppingCart->idshopping_cart);
            $documenttype = $params['documenttype'] ?? $shoppingCart->documenttype;
            $productsShoppingCart = DetailShoppingCart::where('idshopping_cart', $shoppingCart->id)->active();
            $totalProductDiscounts = $productsShoppingCart->sum('discount');
            $grossTotalProducts = $productsShoppingCart->sum('gross_subtotal');
            $shoppingCart->total_discount =  $shoppingCart->global_discount + $totalProductDiscounts;
            $shoppingCart->documenttype = $documenttype;

            $settingBilling = SettingBilling::where('idcompany', $params['idcompany'])->where('documenttype', $documenttype)->first();
            if(!$settingBilling) return $this->errorResponse('La empresa no cuenta con configuracion por empresa', 400);

            #calculo con netos
            if ( $settingBilling->calculationtype  === "1") {
            
                $sub_total = $grossTotalProducts;
                $shoppingCart->sub_total = $sub_total;
            
            }

            #calculo con brutos
            if ( $settingBilling->calculationtype  === "2") {

                $sub_total = ($grossTotalProducts) / (1 + $this->igv);
                $shoppingCart->sub_total = $sub_total;
            }

            $discountGlobalMount = $shoppingCart->discounttype === 2 ? ($sub_total * ($shoppingCart->global_discount/100)) :  $shoppingCart->global_discount;

            $net_total = $sub_total - $discountGlobalMount;
            $shoppingCart->net_total =  $net_total;

            $shoppingCart->IGV = $net_total * $this->igv;
            $shoppingCart->gross_total = $net_total + ($net_total * $this->igv);
            
            $shoppingCart->total = $grossTotalProducts - $shoppingCart->global_discount;
            
            $shoppingCart->save();

            DB::commit();
            return $this->successResponse('Producto actualizado al carrito satisfactoriamente.', $detailShoppingCart);
        } catch (\Throwable $th) {
            throw $th;
            DB::rollBack();
            return $this->externalError('durante el agregado del producto al carrito.', $th->getMessage());
        }

    }

    public function updateProductOld($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del detalle
            $validate = $this->verifyDetailShoppingCart($id);
            if (!$validate->original['status']) return $validate;

            $initialSCProduct = $validate->original['data']['detail'];

            # Traemos todos los campos del producto
            // $productList = DetailPriceListProduct::activeForID($initialSCProduct->idpricelist_product)->first();
            $product = Product::activeForID($initialSCProduct->idproduct)->first();

            // if ($productList) {
            //     $productList = PriceListProductResource::make($productList->load('product', 'pricelist'));
            //     $productList = json_decode(json_encode($productList));
            //     $stock = $productList->stock;
            //     $idregister = $productList->id;
            //     $type = 1;
            //     $price = $productList->price;
            //     $discounttype_pricelist =  $productList->discounttype_pricelist;
            // }else {
                $productStock = StockProduct::where('idproduct', $product->id)->company()->active()->first();
                $stock = $productStock->stock_system ?? 0;
                $idregister = $product->id;
                $type = 2;
                $price = $product->price;
                $discounttype_pricelist = 1; 
            // }
            // if ($stock < $params['quantity']) return $this->errorResponse('No puedes exceder la cantidad de productos que hay en stock', 400);

            $quantity = $params['quantity'] ?? $initialSCProduct->quantity;
            $discount = $initialSCProduct->discount;

            # Calculamos el descuento dado el caso si se manda
            $discount = $params['discount'] ?? $initialSCProduct->discount;

            $validateDiscount = $this->validateDiscount($product, $initialSCProduct->price, $discount, $discounttype_pricelist);
            if (!$validateDiscount->original['status']) return $validateDiscount;
            $calculateDiscount = $validateDiscount->original['data']['detail']; // Se calcular el descuento en dinero

            $checkDiscount = $discount > 0 ? true : false;

            # Se calcula el total bruto y el total descuento del producto
            $grossSubtotal  = $quantity * $initialSCProduct->price;
            $netSubtotal    = $this->calculateGrossSubtotal($quantity, $initialSCProduct->price, $discount, $discounttype_pricelist);

            # Se actualiza los campos del detalle del carrito
            $detailShoppingCart = DetailShoppingCart::find($id);
            $detailShoppingCart->quantity       = $quantity;
            $detailShoppingCart->discount       = $discounttype_pricelist != 3 ? $discount : 0;
            $detailShoppingCart->offer_price    = $discount > 0 ? $product->price - $calculateDiscount : 0;
            $detailShoppingCart->gross_subtotal = $grossSubtotal;
            $detailShoppingCart->net_subtotal   = $netSubtotal;
            $detailShoppingCart->save();

            # Descuento total
            $beforeDiscountTotal = $initialSCProduct->gross_subtotal - $initialSCProduct->net_subtotal;
            $discountTotal = $calculateDiscount * $quantity;

            #Actualizo el total en el carrito
            $shoppingCart = ShoppingCart::find($detailShoppingCart->idshopping_cart);

            $data['gross_total'] = ($shoppingCart->gross_total - $initialSCProduct->gross_subtotal) + $detailShoppingCart->gross_subtotal;
            $data['net_total'] = $data['gross_total'] / (1 + $this->igv);
            $data['IGV'] = $data['net_total'] * $this->igv;

            if ($shoppingCart->global_discount == 0) {
                $data['total_discount'] = ($shoppingCart->total_discount - $beforeDiscountTotal) + $discountTotal;
                $data['total'] = $data['gross_total'] - $data['total_discount'];
            } else {
                if ($checkDiscount) {
                    
                    $data['total_discount'] = $discountTotal + $shoppingCart->total_discount;
                    $data['total'] = $data['gross_total'] - $data['total_discount'];
                } else {
                    $data['total'] = $data['gross_total'] - $shoppingCart->global_discount;
                }
            }

            $updateShoppingCart = ShoppingCart::updateOrCreate(['id' => $shoppingCart->id], $data);
            $updateShoppingCart->fresh();

            DB::commit();
            return $this->successResponse('Producto agregado al carrito satisfactoriamente.', $updateShoppingCart);
        } catch (\Throwable $th) {
            throw $th;
            DB::rollBack();
            return $this->externalError('durante el agregado del producto al carrito.', $th->getMessage());
        }
    }

    public function deleteProduct($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del supervisor
            $validate = $this->verifyDetailShoppingCart($id);
            if (!$validate->original['status']) return $validate;

            $detailShoppingCart = DetailShoppingCart::find($id);
            $detailShoppingCart->update(['status' => 2]);

            #Actualizo el total en el carrito
        
            $shoppingCart = ShoppingCart::find($detailShoppingCart->idshopping_cart);
            $productsShoppingCart = DetailShoppingCart::where('idshopping_cart', $shoppingCart->id)->active();
            $totalProductDiscounts = $productsShoppingCart->sum('discount');
            $grossTotalProducts = $productsShoppingCart->sum('gross_subtotal');

            $settingBilling = SettingBilling::where('idcompany', $params['idcompany'])->where('documenttype', $shoppingCart->documenttype)->first();
            if(!$settingBilling) return $this->errorResponse('La empresa no cuenta con configuracion por empresa', 400);

            if($grossTotalProducts == 0) {
                $shoppingCart->update(['status' => 2]);
            }else {
                
                #calculo con netos
                if ( $settingBilling->calculationtype  === "1") {
                    
                    $sub_total = $grossTotalProducts;
                    $shoppingCart->sub_total = $sub_total;
                
                }

                #calculo con brutos
                if ( $settingBilling->calculationtype  === "2") {

                    $sub_total = ($grossTotalProducts) / (1 + $this->igv);
                    $shoppingCart->sub_total = $sub_total;
                }
                
                $discountGlobalMount = $shoppingCart->discounttype === 2 ? ($sub_total * ($shoppingCart->global_discount/100)) :  $shoppingCart->global_discount;
                $net_total = $sub_total - $discountGlobalMount;
                $shoppingCart->net_total =  $net_total;
    
                $shoppingCart->IGV = $net_total * $this->igv;
                $shoppingCart->gross_total = $net_total + ($net_total * $this->igv);
                
                $shoppingCart->total = $grossTotalProducts - $shoppingCart->global_discount;
            
                $shoppingCart->total_discount = $shoppingCart->global_discount + $totalProductDiscounts;

                $shoppingCart->total = $grossTotalProducts;
                $shoppingCart->save();

            }

            DB::commit();
            return $this->successResponse('Producto eliminado al carrito satisfactoriamente.', $detailShoppingCart);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminiación del producto al carrito.', $th->getMessage());
        }
    }

    public function assignCustomer($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del supervisor
            $validate = $this->verifyShoppingCart($id);
            if (!$validate->original['status']) return $validate;

            $beforeShoppingCart = $validate->original['data']['detail'];

            if ($beforeShoppingCart->iduser_created == $params['iduser_created'] && $beforeShoppingCart->iduser_created == $params['idcustomer']) return $this->errorResponse('El cliente ya esta asignado a un carrito creado por un vendedor', 400);

            $shoppingCart = ShoppingCart::find($id);
            $shoppingCart->idcustomer = $params['idcustomer'];
            $shoppingCart->save();

            DB::commit();
            return $this->successResponse('Cliente asignado/cambiado al carrito satisfactoriamente.', $shoppingCart);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la asignación del cliente al carrito.', $th->getMessage());
        }
    }

    private function verifyShoppingCart($id)
    {
        $shoppingCart = ShoppingCart::activeForID($id)->first();
        if (!$shoppingCart) return $this->errorResponse('El carrito de compras seleccionado no esta disponible', 400);

        return $this->successResponse('OK', $shoppingCart);
    }

    private function verifyDetailShoppingCart($id)
    {
        $detailShoppingCart = DetailShoppingCart::activeForID($id)->first();
        if (!$detailShoppingCart) return $this->errorResponse('El producto seleccionado del carrito no esta disponible', 400);

        return $this->successResponse('OK', $detailShoppingCart);
    }

    private function calculateGrossSubtotal($quantity, $price, $discount, $typeDiscount)
    {
        // 1=$, 2=%, 3=sin descuento
        $discountPrice = $discount;
        if ($typeDiscount == 2) $discountPrice = $price * ($discount / 100);

        if ($typeDiscount == 3) $discountPrice = 0;

        $subtotal = ($price - $discountPrice) * $quantity;

        return $subtotal;
    }

    private function validateDiscount($product, $price, $discount, $typeDiscount)
    {
        // 1=$, 2=%, 3=sin descuento
        $discountPrice = $discount;
        $maxDiscount = 0;

        if ($typeDiscount == 3) return $this->successResponse('OK', '0');

        if (isset($product->discount) && $product->discount > 0) {
            $maxDiscount = $product->discount;
        } else if (isset($product->max_desc_item_pricelist) && $product->max_desc_item_pricelist > 0) {
            $maxDiscount = $product->max_desc_item_pricelist;
        } 

        if ( $maxDiscount > 0 && $discount > $maxDiscount) return $this->errorResponse('El descuento sobrepasada el limite impuesto', 400);

        if ($typeDiscount == 2) $discountPrice = $price * ($discount / 100);

        return $this->successResponse('OK', $discountPrice);
    }
}
