<?php

namespace App\Services;

use App\Http\Resources\SaleChannelResource;
use App\Models\Company;
use App\Models\SaleChannel;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class SaleChannelService
 * @package App\Services
 */
class SaleChannelService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $channels = SaleChannel::channelFilters()->orderBy('id', 'desc');

        $channels = !empty($withPagination)
            ? $channels->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $channels->get();

        $channels = SaleChannelResource::collection($channels->load('company'));

        return $this->successResponse('Lectura exitosa.', $channels);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            #Validar empresa
            $company = Company::activeForID($params['idcompany'])->first();
            if (!$company) return $this->errorResponse('La empresa seleccionada no es válida.', 400);

            #Validar duplicidad de nombre
            $name = SaleChannel::where('name', $params['name'])->active()->company()->first();
            if ($name) return $this->errorResponse('El nombre ingresado ya se encuentra registrado.', 400);

            $channel = SaleChannel::create([
                'name' => $params['name'],
                'description' => $params['description'] ?? null,
                'idcompany' => $params['idcompany']
            ]);
            $channel->save();
    
            DB::commit();
            return $this->successResponse('Canal de venta registrado satisfactoriamente.', $channel);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un canal de venta.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            #Validar empresa
            $company = Company::activeForID($params['idcompany'])->first();
            if (!$company) return $this->errorResponse('La empresa seleccionada no es válida.', 400);

            #Validar canal de venta
            $channel = SaleChannel::activeForID($id)->company()->first();
            if (!$channel) return $this->errorResponse('Canal de venta no válido.', 400);

            #Validar duplicidad de nombre
            $name = SaleChannel::where('id', '!=', $id)->where('name', $params['name'])->company()->active()->first();
            if ($name) return $this->errorResponse('El nombre ingresado ya se encuentra registrado.', 400);

            $channel->update($params);
            $channel->save();

            DB::commit();
            return $this->successResponse('Canal de venta actualizado satisfactoriamente.', $channel);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un canal de venta.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            #Validar canal de venta
            $channel = SaleChannel::activeForID($id)->company()->first();
            if (!$channel) return $this->errorResponse('Canal de venta no válido.', 400);

            $channel->status = 2;
            $channel->save();

            DB::commit();
            return $this->successResponse('Canal de venta eliminado satisfactoriamente.', $channel);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un canal de venta.', $th->getMessage());
        }
    }
}
