<?php

namespace App\Services;

use App\Exports\ProductsTemplateExport;
use App\Exports\ProductsTemplateInvetoryExport;
use App\Exports\ProductsTemplateNotInvetoryExport;
use App\Http\Resources\ProductResource;
use App\Imports\ProductsDataImport;
use App\Imports\ProductsDataImportInventory;
use App\Imports\ProductsDataImportNotInventory;
use App\Models\Category;
use App\Models\Company;
use App\Models\CompanyConfiguration;
use App\Models\DetailPriceListProduct;
use App\Models\PriceList;
use App\Models\Product;
use App\Models\SpecificUse;
use App\Models\StockProduct;
use App\Models\TypeProduct;
use App\Models\UsageAttribute;
use App\Traits\HasResponse;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Crypt;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Maatwebsite\Excel\Facades\Excel;
use Tymon\JWTAuth\Facades\JWTAuth;
use Illuminate\Support\Facades\Log;

class ProductService
{
    use HasResponse;

    /** @var SoftnetService */
    private $softnetService;

    public function __construct(SoftnetService $softnetService)
    {
        $this->softnetService = $softnetService;
    }

    public function list($withPagination, $params)
    {
        $products = Product::orderBy('id', 'desc');
        // productFilters()
            // ->orderBy('id', 'desc');

        if(isset($params['idcategory'])) {
            $products = $products->where('idcategory', $params['idcategory']);
        }

        if(isset($params['idsubcategory'])) {
            $products = $products->where('idsubcategory', $params['idsubcategory']);
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $products->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%")
                ->orWhere('code_erp', 'like', "%$search%")
                ->orWhere('code_system', 'like', "%$search%");
            });
        }

        $products = !empty($withPagination)
            ? $products->paginate($withPagination['perPage'], ['*'], 'page', $withPagination['page'])
            : $products->get();

        $products = ProductResource::collection($products->load(
            'stock', 'company', 'cointType', 'category', 'type', 'supplier', 'unitmeasure', 'businessarea', 'classification'
        ));

        return $this->successResponse('Lectura exitosa.', $products);
    }

    public function listByCompany($params, $withPagination)
    {
        $products = Product::where('status', 1)->orderBy('id', 'desc');

        $idcompany = isset($params['idcompany']) ? $params['idcompany'] : '';
        if($idcompany) {
            $products->where(function ($query) use ($idcompany) {
                $query->where('idcompany', $idcompany);
            });
        }

        $search = isset($params['search']) ? $params['search'] : '';
        if($search){
            $products->where(function ($query) use ($search) {
                $query->where('name', 'like', "%$search%")
                ->orWhere('code_system', 'like', "%$search%");
            });
        }

        $products = !empty($withPagination)
            ? $products->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $products->get();

        $products = ProductResource::collection($products->load('company', 'cointType', 'category', 'subcategory', 'specificsubcategory', 'type', 'supplier', 'unitmeasure', 'businessarea', 'classification'));

        return $this->successResponse('Lectura exitosa.', $products);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            //Validar si tiene code_system o code_erp
            if (isset($params['code_erp'])) {
                $validate = $this->validateCode($params['code_erp'], 2);

                $params['code_system'] = $this->generateCodeSystem($params['code_erp'], $params['idtype']);
            } else {
                $validate = $this->validateCode($params['code_system'], 1);
            }

            if (!$validate->original['status']) return $validate;

            $imageFields = ['main_image', 'optional_image1', 'optional_image2', 'optional_image3', 'optional_image4'];
            foreach ($imageFields as $field) {
                if (isset($params[$field])) $params[$field] = $this->saveImage($params[$field], $params['code_system'], $params['idcompany']);
            }

            $product = Product::create($params);
            $product->fresh();

            # Agregamos un producto mas a la categoria
            if(isset($params['idcategory']) && $params['idcategory']){
                $category = Category::where('id', $params['idcategory'])->first();
                $category->quantity_products = $category->quantity_products + 1;
                $category->save();
            }

            # Agregamos el producto al listado de precio general de la compañia
            $priceLists = PriceList::where('type', 1)->company()->active()->get();
            foreach ($priceLists as $priceList) {
                $stock = StockProduct::where('idproduct', $product->id)->active()->company()->sum('stock_system');

                $detail = DetailPriceListProduct::create([
                    'idpricelist' => $priceList->id,
                    'price' => $product->price,
                    'cost' => $product->cost,
                    'stock' => $stock,
                    'subtotal' => $product->price * $stock,
                    'discount' => 0,
                    'idproduct' => $product->id,
                    'idcompany' => $params['idcompany'],
                ]);
                $detail->fresh();
            }

            DB::commit();
            return $this->successResponse('Producto creado satisfactoriamente.', $product);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un producto.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del producto
            $validate = $this->verifyProduct($id);
            if (!$validate->original['status']) return $validate;

            $product = $validate->original['data']['detail'];

            //Validar si tiene code_system o code_erp
            if (isset($params['code_erp'])) {
                $validate = $this->validateCode($params['code_erp'], 2, $id);

                $params['code_system'] = $this->generateCodeSystem($params['code_erp'], $params['idtype']);
            } else {
                $validate = $this->validateCode($params['code_system'], 1, $id);
            }

            if (!$validate->original['status']) return $validate;

            $imageFields = ['main_image', 'optional_image1', 'optional_image2', 'optional_image3', 'optional_image4'];
            foreach ($imageFields as $field) {
                if (isset($params[$field])) $params[$field] = $this->saveImage($params[$field], $params['code_system'], $params['idcompany']);
            }

            $product = Product::find($id);
            $product->update($params);
            $product->save();

            DB::commit();
            return $this->successResponse('Producto actualizado satisfactoriamente.', $product);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un producto.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez de producto
            $validate = $this->verifyProduct($id);
            if (!$validate->original['status']) return $validate;

            $product = Product::find($id);
            $product->status = 2;
            $product->save();

            $category = Category::where('id', $product->idcategory)->first();
            $category->quantity_products = $category->quantity_products - 1;
            $category->save();

            DB::commit();
            return $this->successResponse('Producto eliminado satisfactoriamente.', $product);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un producto.', $th->getMessage());
        }
    }

    private function verifyProduct($idproduct)
    {
        $product = Product::activeForID($idproduct)->first();
        if (!$product) return $this->errorResponse('No se encontró el producto.', 400);

        return $this->successResponse('OK', $product);
    }

    private function saveImage($img, $code, $idcompany, $old_file = null)
    {
        try {
            $company = Company::activeForID($idcompany)->first();
            $currentDateTime = Carbon::now()->format('Ymd_His');
            $extension = $img->getClientOriginalExtension();
            $name = "main_image_" . JWTAuth::user()->id . "_" . $currentDateTime.$extension;
            $fileImg = "public/pdf/$company->rut/productos/$code";

            if (isset($old_file)) {
                // Si existe, eliminarlo antes de guardar la nueva imagen
                if (Storage::exists($old_file)) Storage::delete($old_file);
            }

            //Guardamos la imagen
            $image = $img->storeAs($fileImg, $name);
            return $this->successResponse('OK', $image);
        } catch (\Throwable $th) {
            return $this->externalError('durante la creación de un producto.', $th->getMessage());
        }
    }

    private function validateCode($code, $typeCode, $id = null)
    {
        try {
            $field = $typeCode == '1' ? 'code_system' : 'code_erp';

            $product = Product::code($code, $field, $id ?? null)->active()->first();

            if($product) return $this->errorResponse('Ya se encuentra registrado el código.', 400);

            return $this->successResponse('OK', $product);
        } catch (\Throwable $th) {
            return $this->externalError('durante la validación del código.', $th->getMessage());
        }
    }

    private function generateCodeSystem($code, $type)
    {
        $codeProduct = Product::selectRaw('MAX(code_system) as max_code_system')->where('idtype', $type)->whereNotNull('code_erp')->active()->first();
        $type = TypeProduct::find($type);

        if ($codeProduct) {
            // Obtener el número de la parte numérica del código existente
            $counter = (int) substr($codeProduct->max_code_system, -9);
            $code = substr($codeProduct->max_code_system, 0, -9);

            // Sumar 1 al número existente
            $counter += 1;

            // Formatear el nuevo número a 9 dígitos
            $numberFormat = str_pad($counter, 9, '0', STR_PAD_LEFT);

            return $code . $numberFormat;
        } else {
            return "ERP-$type->code" . '000000001';
        }
    }

    public function downloadTemplate($params)
    {
        $fileName = "products_template.xlsx";
        $baseUrl = url('/');  // Esto generará la URL base de la aplicación

        $config = CompanyConfiguration::where('idcompany', $params['idcompany'])->active()->first();

        if($config->inventory_configuration == '1') {
            Excel::store(new ProductsTemplateNotInvetoryExport([]), $fileName , 'public');
        } else {
            Excel::store(new ProductsTemplateInvetoryExport([]), $fileName , 'public');
        }

        // Excel::store(new ProductsTemplateExport([]), $fileName , 'public');

        return $this->successResponse('OK' , $baseUrl . '/storage/' . $fileName);
    }

    public function importProductExcel($params)
    {
        DB::beginTransaction();
        try {
            $file = $params['products_import'];

            $config = CompanyConfiguration::where('idcompany', $params['idcompany'])->active()->first();

            // Importar datos usando la clase de importación
            if($config->inventory_configuration == '1') Excel::import(new ProductsDataImportNotInventory, $file);

            if($config->inventory_configuration == '2') Excel::import(new ProductsDataImportInventory, $file);

            // Excel::import(new ProductsDataImport, $file);

            DB::commit();
            return $this->successResponse('OK' , 'Datos importados correctamente.');
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
            return $this->externalError('durante la importación de Productos.', $th->getMessage());
        }
    }

    public function listErpProducts($params)
    {
        try {
            #Obtener el token del usuario erp de la empresa
            // $userErp = UserERP::where('id', $params['idcompany'])->first(); #descomentar
            $userErp = (object)[
                'rut'                => '22222222-2',
                'user'               => 'usuario',
                'password'           => 'demoerp',
                'encrypted_password' => Crypt::encryptString('demoerp'),
                'status' => 1
            ];
            if (!$userErp) return $this->errorResponse('La empresa no registra su usuario del ERP para comenzar a facturar.', 400);
            $token    = $this->softnetService->getTokenSoftnet($userErp);
            // $tokenErp =  Crypt::encryptString($token);
            $tokenErp =  $token;

            $params['token_erp'] = $tokenErp;
            $erpProducts = $this->softnetService->getProductos($params)->original['data']['detail'];

            return $this->successResponse('Lecura exitosa.', $erpProducts);
        } catch (\Throwable $th) {
            throw $th;
        }
    }
}
