<?php

namespace App\Services;

use App\Http\Resources\ProductClassificationResource;
use App\Models\BranchOffice;
use App\Models\Company;
use App\Models\ProductClassification;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class ProductClassificationService
 * @package App\Services
 */
class ProductClassificationService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $classifications = ProductClassification::clasificationFilters()->orderBy('id', 'desc');

        $classifications = !empty($withPagination)
            ? $classifications->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $classifications->get();

        $classifications = ProductClassificationResource::collection($classifications->load('branchOffice', 'company'));

        return $this->successResponse('Lectura exitosa.', $classifications);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            #Validar empresa
            $company = Company::activeForID($params['idcompany'])->first();
            if (!$company) return $this->errorResponse('La empresa seleccionada no es válida.', 400);

            #Validar sucursal
            $branch = BranchOffice::activeForID($params['idbranchoffice'])->company()->first();
            if (!$branch) return $this->errorResponse('La sucursal seleccionada no es válida.', 400);

            #Validar duplicidad de nombre
            $name = ProductClassification::where('name', $params['name'])->active()->company()->first();
            if ($name) return $this->errorResponse('El nombre ingresado ya se encuentra registrado.', 400);

            $classification = ProductClassification::create([
                'name' => $params['name'],
                'idbranchoffice' => $params['idbranchoffice'],
                'idcompany' => $params['idcompany']
            ]);
            $classification->save();
    
            DB::commit();
            return $this->successResponse('Clasificación de producto registrada satisfactoriamente.', $classification);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de una clasificación de producto.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            #Validar clasificación
            $classification = ProductClassification::activeForID($id)->first();
            if (!$classification) return $this->errorResponse('Clasificación no válida.', 400);

            #Validar empresa
            $company = Company::activeForID($params['idcompany'])->first();
            if (!$company) return $this->errorResponse('La empresa seleccionada no es válida.', 400);

            #Validar sucursal
            $branch = BranchOffice::activeForID($params['idbranchoffice'])->company()->first();
            if (!$branch) return $this->errorResponse('La sucursal seleccionada no es válida.', 400);

            #Validar duplicidad de nombre
            $name = ProductClassification::where('id', '!=', $id)->where('name', $params['name'])->where('idbranchoffice', $params['idbranchoffice'])->company()->active()->first();
            if ($name) return $this->errorResponse('El nombre ingresado ya se encuentra registrado.', 400);

            $classification->update($params);
            $classification->save();

            DB::commit();
            return $this->successResponse('Clasificación de producto registrada satisfactoriamente.', $classification);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de una clasificación de producto.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            #Validar tipo de produto
            $type = ProductClassification::activeForID($id)->first();
            if (!$type) return $this->errorResponse('Clasificación de producto no válido.', 400);

            $type->status = 2;
            $type->save();

            DB::commit();
            return $this->successResponse('Clasificación de producto eliminada satisfactoriamente.', $type);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de una clasificación de producto.', $th->getMessage());
        }
    }
}
