<?php

namespace App\Services;

use App\Http\Resources\PermissionsUserResource;
use App\Models\DetailUserCompany;
use App\Models\Permissions;
use App\Models\PermissionsUser;
use App\Models\User;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

class PermissionsUserService
{
    use HasResponse;

    public function list($withPagination)
    {
        $permissionsUser = PermissionsUser::permissionsUserFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc');

        $permissionsUser = !empty($withPagination)
            ? $permissionsUser->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $permissionsUser->get();

        $permissionsUser = PermissionsUserResource::collection($permissionsUser->load('user'));

        return $this->successResponse('Lectura exitosa.', $permissionsUser);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            # Verificar duplicidad de usuario
            $permissionsUserExisting = PermissionsUser::where('iduser', $params['iduser'])->first();
            if ($permissionsUserExisting) return $this->errorResponse('El usuario seleccionado ya cuenta con un registro vigente.', 400);

            # Verificar que no sea un usuario superadmin y que los permisos asignados existan
            $validate = $this->verifyFK($params);
            if (!$validate->original['status']) return $validate;

            $permissionsUser = PermissionsUser::create([
                'iduser'        => $params['iduser'],
                'idcompany'     => $params['idcompany'],
                'permissions'   => $params['permissions'],
                'actions'       => $params['actions'],
            ]);
            $permissionsUser->fresh();

            DB::commit();
            return $this->successResponse('Usuario con permisos personalizados creado satisfactoriamente.', $permissionsUser);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la asignacion de permisos personalizados para un usuario.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del conductor
            $validate = $this->checkRegister($id);
            if (!$validate->original['status']) return $validate;

            # Verificar que no sea un usuario superadmin y que los permisos asignados existan
            $validate = $this->verifyFK($params);
            if (!$validate->original['status']) return $validate;

            $permissionsUser = PermissionsUser::find($id);
            $permissionsUser->update($params);

            DB::commit();
            return $this->successResponse('Usuario con permisos personalizados actualizado satisfactoriamente.', $permissionsUser);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de una asignación de permisos personalizados de usuarios.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del conductor
            $validate = $this->checkRegister($id);
            if (!$validate->original['status']) return $validate;

            $permissionsUser = PermissionsUser::find($id);
            $permissionsUser->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('El usuario ya no cuenta con permisos personalizados.', $permissionsUser);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación permisos personalizados para un usuario.', $th->getMessage());
        }
    }

    private function checkRegister($id)
    {
        $permissionsUser = PermissionsUser::activeForID($id)->first();
        if (!$permissionsUser) return $this->errorResponse('El registro de personalización de permisos es inválido.', 400);

        return $this->successResponse('OK');
    }

    private function verifyFK($params)
    {
        if (isset($params['iduser'])) {
            # Verificar que no sea un usuario tipo superadmin
            $user = User::find($params['iduser']);
            if (!$user) return $this->errorResponse('El usuario seleccionado es inválido.', 400);

            $detailUser = DetailUserCompany::where('iduser_type', $user->id)
                ->where('idcompany', $params['idcompany'])->first();

            if (!$detailUser || $detailUser->iduser_type == 1) return $this->errorResponse('El usuario seleccionado es inválido.', 400);
        }

        if (isset($params['permissions'])) {
            # Verificar que los permisos asignados sean válidos
            $validatePermissions = Permissions::whereIn('id', $params['permissions'])->count();
            if (count($params['permissions']) !=  $validatePermissions) return $this->errorResponse('Selección de permisos inválida.', 400);
        }

        if (isset($params['actions'])) {
            # Verificar que los permisos asignados sean válidos
            $repeat = count($params['actions']) === count(array_unique($params['actions']));
            if (!$repeat) return $this->errorResponse('Las acciones seleccionadas son inválidas.', 400);
        }


        return $this->successResponse('OK');
    }
}
