<?php

namespace App\Services;

use App\Http\Resources\MembershipCostsResource;
use App\Models\Membership;
use App\Models\MembershipCosts;
use App\Models\MembershipPlans;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

class MembershipCostsService
{
    use HasResponse;

    public function getCosts()
    {
        $costs = MembershipCosts::active()->get()->load('plans', 'membership');
        $costs = MembershipCostsResource::collection($costs);
        return $this->successResponse('Lectura exitosa.', $costs);
    }

    public function getCostsById($idcost)
    {
        $costs = MembershipCosts::where('id', $idcost)->active()->get()->load('plans', 'membership');
        $costs = MembershipCostsResource::collection($costs);
        return $this->successResponse('Lectura exitosa.', $costs);
    }

    public function getCostsTreatment()
    {
        $costs = MembershipCosts::active()->get()->load('plans', 'membership');
        $costs = MembershipCostsResource::collection($costs);

        $costs = json_decode(json_encode($costs));

        $memberships =  array_unique(array_column($costs, 'membership'));

        $arrayMembership = collect($memberships)->map(function ($membershipName) {
            $dataMembership = Membership::where('name', $membershipName)->first();

            return $dataMembership ? (object)[
                'id' => $dataMembership->id,
                'membership' => $membershipName,
                'plans' => [],
            ] : null;
        })->filter();

        $costsByMembership = collect($costs)->groupBy('idmembership');

        collect($arrayMembership)->each(function ($membership) use ($costsByMembership) {
            $membership->plans = $costsByMembership->get($membership->id, collect())->toArray();
        });

        return $this->successResponse('Lectura exitosa.', $arrayMembership);
    }

    public function getCostsAll()
    {
        $costs = MembershipCosts::whereIn('status', [1, 3])->get()->load('plans', 'membership');
        $costs = MembershipCostsResource::collection($costs);
        return $this->successResponse('Lectura exitosa.', $costs);
    }

    public function createCosts($params)
    {
        DB::beginTransaction();
        try {
            $validate = $this->validateCombo($params);
            if (!$validate->original['status']) return $validate;

            $membershipCosts = MembershipCosts::create([
                'idplans'       => $params['idplans'],
                'idmembership'  => $params['idmembership'],
                'amount'        => $params['amount'],
                'date_start'    => $params['date_start'],
                'date_end'      => $params['date_end'],
                'cod_erp'       => $params['cod_erp']
            ]);
            $membershipCosts->fresh();

            DB::commit();
            return $this->successResponse('Costos de membresias creado con éxito', $membershipCosts);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de una membresía.', $th->getMessage());
        }
    }

    public function editDateCosts($id, $params)
    {
        DB::beginTransaction();
        try {
            $validate = $this->validateMembershipCosts($id);
            if (!$validate->original['status']) return $validate;


            $membershipCosts = MembershipCosts::updateOrCreate(
                ['id' => $id],
                [
                    'date_end'  => $params['date_end'],
                    'UF'        => $params['UF']
                ]
            );
            $membershipCosts->fresh();

            DB::commit();
            return $this->successResponse('Fecha editada con éxito.', $membershipCosts);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la edición de una membresía.', $th->getMessage());
        }
    }

    public function deleteCosts($id)
    {
        DB::beginTransaction();
        try {
            $validate = $this->validateMembershipCosts($id);
            if (!$validate->original['status']) return $validate;

            $membershipCosts = MembershipCosts::updateOrCreate(
                ['id' => $id],
                ['status' => 2]
            );
            $membershipCosts->fresh();

            DB::commit();
            return $this->successResponse('Costo de membresía eliminada con éxito.');
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function validateCombo($params)
    {
        $plans = MembershipPlans::find($params['idplans']);
        if (!$plans) return $this->errorResponse('El id del plan de la membresia no existe', 400);

        $membership = Membership::find($params['idmembership']);
        if (!$membership) return $this->errorResponse('El id de la membresia no existe', 400);


        if (isset($params['date_end']) && $params['date_start'] > $params['date_end']) {
            return $this->errorResponse('La fecha fin no puede ser mayor que la fecha de inicio', 400);
        }

        return $this->successResponse('OK');
    }

    public function validateMembershipCosts($id)
    {
            $membershipCosts = MembershipCosts::find($id);
            if (!isset($membershipCosts)) {
                return $this->errorResponse('No existe este Costo de Membresia', 400);
            }

            return $this->successResponse('OK');

    }
}
