<?php

namespace App\Services;

use App\Http\Resources\DetInitialStepResource;
use App\Http\Resources\InitialStepResource;
use App\Models\DetInitialStep;
use App\Models\InitialStep;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;

/**
 * Class InitialStepService
 * @package App\Services
 */
class InitialStepService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $steps = InitialStep::initialStepFilters();

        $steps = !empty($withPagination)
            ? $steps->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $steps->get();

        $steps = InitialStepResource::collection($steps->load('detInitialStep'));

        return $this->successResponse('Lectura exitosa.', $steps);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            #Validar duplicidad de descripción
            $description = InitialStep::where('description', $params['description'])->active()->first();
            if ($description) return $this->errorResponse('Ya existe un registro con la descripción ingresada.', 400);

            $step = InitialStep::create([
                'description' => $params['description']
            ]);

            #Agregar el id creado al plan seleccionado
            if (isset($params['plans'])) {
                $details = [];

                foreach ($params['plans'] as $typePlan) {
                    $order = DetInitialStep::where('key_plan', $typePlan)->active()->count();
                    $details[] = [
                        'idinitial_step'    => $step->id,
                        'key_plan'          => $typePlan,
                        'order'             => $order + 1
                    ];
                }

                DetInitialStep::insert($details);
            }

            DB::commit();
            return $this->successResponse('Registrado con éxito.', $step);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            #Validar duplicidad de descripción
            $description = InitialStep::where('id', $id)->where('description', $params['description'])->active()->first();
            if ($description) return $this->errorResponse('Ya existe un registro con la descripción ingresada.', 400);

            $step = InitialStep::activeForId($id)->first();
            $step->update($params);

            #Editar relaciones con energías
            if (isset($params['plans'])) {
                $insertsPlans = [];
                foreach ($params['plans'] as $typePlan) {
                    $detail = DetInitialStep::where('idinitial_step', $id)->where('key_plan', $typePlan)->active()->first();
                    if (!$detail) {
                        $insertsPlans[] = [
                            'idinitial_step' => $id,
                            'key_plan' => $typePlan
                        ];
                    }
                }
                if (count($insertsPlans) > 0) DetInitialStep::insert($insertsPlans);

                #Eliminar relaciones
                $stepDet = DetInitialStep::where('idinitial_step', $id)->active();
                if (!empty($params['plans'])) $stepDet->whereNotIn('key_plan', $params['plans']);
                $stepDet->update(['status' => '2']);
            }

            DB::commit();
            return $this->successResponse('Actualizado satisfactoriamente.', $step);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la edición.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        try {
            #Validar duplicidad de descripción
            $step = InitialStep::activeForId($id)->first();
            if (!$step) return $this->errorResponse('Paso inicial no válido.', 400);


            //Corregir los ordenes en el detalle
            #Obtener los planes a los que pertenece el paso inicial
            $details = DetInitialStep::where('idinitial_step', $id)->active()->pluck('key_plan');

            if (count($details) > 0) {
                foreach ($details as $value) {
                    #Detalle correspondiente al plan e id
                    $detStep = DetInitialStep::where('idinitial_step', $id)->where('key_plan', $value)->active()->first();

                    #Tamaño del detalle
                    $count = DetInitialStep::where('key_plan', $value)->active()->count();
                    
                    #Si el tamaño del detalle es 1 o 0 omitir los siguientes pasos
                    if ($count < 2) continue;

                    #Si es el último no hay nada por corregir
                    if ($detStep->order == $count) continue;

                    #Ordenar los pasos siguientes al eliminado
                    if ($detStep->order < $count) {
                        DetInitialStep::where('order', '>', $detStep->order)->active()->update([
                            'order' => DB::raw('`order` - 1')
                        ]);
                    }
                }
            }

            $step->update(['status' => 2]);
            DetInitialStep::where('idinitial_step', $id)->active()->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('Eliminado satisfactoriamente.', $step);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación.', $th->getMessage());
        }
    }

    public function listDetail($params, $withPagination)
    {
        $steps = DetInitialStep::detInitialStepFilters()->orderBy('order', 'asc');

        $steps = !empty($withPagination)
            ? $steps->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $steps->get();

        $steps = DetInitialStepResource::collection($steps->load('initialStep'));

        return $this->successResponse('Lectura exitosa.', $steps);
    }

    public function changeOrder($params)
    {
        DB::beginTransaction();
        try {
            $validation = Validator::make($params, [
                'iddet_initial_setp' => 'required|numeric|validate_ids_exist:DetInitialStep',
                'order'              => 'required|string|in:up,down'
            ]);
            if ($validation->fails()) return $this->errorResponse($validation->errors(), 400);

            $step = DetInitialStep::where('id', $params['iddet_initial_setp'])->first();

            if ($params['order'] == 'up') {
                #Validar que no sea el primero
                if ($step->order == 1) {
                    return $this->errorResponse('El paso inicial ya tiene la prioridad máxima.', 400);
                }

                #Obtener el paso anterior
                $beforeOrder = (int) $step->order - 1;
                $beforeStep = DetInitialStep::where('key_plan', $step->key_plan)->where('order', $beforeOrder)->active()->first();

                $beforeStep->update(['order' => $beforeOrder + 1]);
                $step->update(['order' => $beforeOrder]);
            } else {
                #Validar que no sea el último
                $count = DetInitialStep::where('key_plan', $step->key_plan)->active()->count();
                if ($step->order == $count) {
                    return $this->errorResponse('El paso inicial ya tiene la menor prioridad posible.', 400);
                }

                #Obtener el paso siguiente
                $beforeOrder = (int) $step->order + 1;
                $beforeStep = DetInitialStep::where('key_plan', $step->key_plan)->where('order', $beforeOrder)->active()->first();

                $beforeStep->update(['order' => $beforeStep->order - 1]);
                $step->update(['order' => $step->order + 1]);
            }

            DB::commit();
            return $this->successResponse('Cambio exitoso');
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }
}
