<?php

namespace App\Services;

use App\Models\Import;
use App\Traits\HasResponse;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;

/**
 * Class ImportService
 * @package App\Services
 */
class ImportService
{
    use HasResponse;

    public function list($params, $withPagination)
    {
        $imports = Import::importFilters()->orderBy('id', 'asc');

        $imports = !empty($withPagination)
            ? $imports->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $imports->get();

        return $this->successResponse('Lectura exitosa.', $imports);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            #Verificar duplicidad de descripcion
            $validate = Import::where('description', $params['description'])->active()->first();
            if ($validate) return $this->errorResponse('Ya existe un importador con la descripción ingresada.', 400);

            $xls = $params['document']->store('public/imports');

            $import = Import::create([
                'description' => $params['description'],
                'xls'         => $xls
            ]);

            DB::commit();
            return $this->successResponse('Importación creada exitosamente', $import);
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            #Validar import
            $import = Import::activeForID($id)->first();
            if (!$import) return $this->errorResponse('Importación seleccionada no válida.', 400);

            #Verificar duplicidad de descripcion
            if (isset($params['description'])) {
                $validate = Import::where('id', '!=', $id)->where('description', $params['description'])->active()->first();
                if ($validate) return $this->errorResponse('Ya existe un importador con la descripción ingresada.', 400);
            }

            if (isset($params['document'])) {
                #Eliminar si existe xls
                if (!empty($import->xls)) Storage::delete($import->xls);

                #Almacenar el xls
                $xls = $params['document']->store('public/imports');

                $import->update(['xls' => $xls]);
            }

            $import->update($params);

            DB::commit();
            return $this->successResponse('Importación actualizada exitosamente', $import);
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            #Validar import
            $import = Import::activeForId($id)->first();
            if (!$import) return $this->errorResponse('Importación seleccionada no válida.', 400);

            $import->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('Importación eliminado exitosamente', $import);
        } catch (\Throwable $th) {
            DB::rollBack();
            throw $th;
        }
    }

    public function downloadTemplate($id)
    {
        try {
            #Validar import
            $import = Import::activeForId($id)->first();
            if (!$import || !$import->xls) return $this->errorResponse('Importación seleccionada no válida.', 400);

            if (!Storage::exists($import->xls)) return $this->errorResponse('Error al descargar', 400);

            // Generar la URL de descarga
            $downloadUrl = Storage::url($import->xls);

            return $this->successResponse('Ok', $downloadUrl);
        } catch (\Throwable $th) {
            throw $th;
        }
    }
}
