<?php

namespace App\Services;

use App\Http\Resources\DriversResource;
use App\Models\Drivers;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class DriversService
 * @package App\Services
 */
class DriversService
{
    use HasResponse;

    public function list($withPagination)
    {
        $drivers = Drivers::driverFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc');

        $drivers = !empty($withPagination)
            ? $drivers->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $drivers->get();

        $drivers = DriversResource::collection($drivers->load('company'));

        return $this->successResponse('Lectura exitosa.', $drivers);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            # Verificar duplicidad de RUT
            $rutExisting = Drivers::rut($params['rut'])->first();
            if ($rutExisting) return $this->errorResponse('El rut ya se encuentra registrado.', 400);

            $driver = Drivers::create([
                'rut'           => $params['rut'],
                'name'          => $params['name'],
                'surname'       => $params['surname'],
                'email'         => $params['email'],
                'phone'         => $params['phone'],
                'idcompany'     => $params['idcompany']
            ]);
            $driver->fresh();

            # Crear usuario por Observer

            DB::commit();
            return $this->successResponse('Conductor creado satisfactoriamente.', $driver);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un conductor.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del conductor
            $validate = $this->verifyDriver($id, $params['rut']);
            if (!$validate->original['status']) return $validate;

            $driver = Drivers::find($id);
            $driver->update($params);

            DB::commit();
            return $this->successResponse('Conductor actualizado satisfactoriamente.', $driver);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un conductor.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            # Verificar validez del conductor
            $validate = $this->verifyDriver($id);
            if (!$validate->original['status']) return $validate;

            $driver = Drivers::find($id);
            $driver->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('Conductor eliminado satisfactoriamente.', $driver);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un conductor.', $th->getMessage());
        }
    }

    private function verifyDriver($id, $rut = null)
    {
        $driver = Drivers::activeForID($id)->first();
        if (!$driver) return $this->errorResponse('El conductor seleccionado no esta disponible', 400);

        if(isset($rut)){
            $validateRut = Drivers::rut($rut, $id)->where('status', 1)->first();
            if ($validateRut) return $this->errorResponse('El rut ya se encuentra registrado.', 400);
        }

        return $this->successResponse('OK');
    }
}
