<?php

namespace App\Services;

use App\Http\Resources\AdminCompanyResource;
use App\Models\AdminCompany;
use App\Models\User;
use App\Traits\HasResponse;
use Illuminate\Support\Facades\DB;

/**
 * Class AdminCompanyService
 * @package App\Services
 */
class AdminCompanyService
{
    use HasResponse;

    /** @var UserService */
    private $userService;

    public function __construct(UserService $userService)
    {
        $this->userService = $userService;
    }

    public function list($withPagination)
    {
        $adminCompanies = AdminCompany::adminFilters() # Filtrado por el modelo
            ->orderBy('id', 'desc');

        $adminCompanies = !empty($withPagination)
            ? $adminCompanies->paginate($withPagination['perPage'], page: $withPagination['page'])
            : $adminCompanies->get();

        $adminCompanies = AdminCompanyResource::collection($adminCompanies->load('company'));

        return $this->successResponse('Lectura exitosa.', $adminCompanies);
    }

    public function register($params)
    {
        DB::beginTransaction();
        try {
            # Verificar duplicidad de RUT
            $rutExisting = AdminCompany::rut($params['rut'])->first();
            if ($rutExisting) return $this->errorResponse('El rut ya se encuentra registrado.', 400);

            #Validar duplicidad de username
            if (isset($params['username'])) {
                $username = User::where('username', $params['username'])->active()->company()->first();
                if ($username) return $this->errorResponse('El username ingresado ya se encuentra registrado', 400);
            }

            $adminCompany = AdminCompany::create([
                'rut'           => $params['rut'],
                'name'          => $params['name'],
                'surname'       => $params['surname'],
                'email'         => $params['email'],
                'phone'         => $params['phone'] ?? null,
                'address'       => $params['address'] ?? null,
                'birth_date'    => $params['birth_date'] ?? null,
                'idcompany'     => $params['idcompany']
            ]);
            $adminCompany->fresh();

            $this->userService->registerUser($params['rut'], $params['email'], 2, $params['idcompany'], $params['username'] ?? null);

            DB::commit();
            return $this->successResponse('Administrador creado satisfactoriamente.', $adminCompany);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la creación de un administrador.', $th->getMessage());
        }
    }

    public function update($id, $params)
    {
        DB::beginTransaction();
        try {
            #Validar vendedor
            $adminCompany = AdminCompany::activeForID($id)->first();
            if (!$adminCompany) return $this->errorResponse('El administrador seleccionado no esta disponible', 400);

            #Validar duplicidad de rut
            $rut = AdminCompany::rut($params['rut'], $id)->active()->first();
            if ($rut) return $this->errorResponse('El rut ingresado ya se encuentra registrado', 400);

            $adminCompany->update($params);

            DB::commit();
            return $this->successResponse('Administrador actualizado satisfactoriamente.', $adminCompany);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la actualización de un administrador.', $th->getMessage());
        }
    }

    public function delete($id)
    {
        DB::beginTransaction();
        try {
            #Validar vendedor
            $adminCompany = AdminCompany::activeForID($id)->first();
            if (!$adminCompany) return $this->errorResponse('El administrador seleccionado no esta disponible', 400);

            $adminCompany = AdminCompany::find($id);
            $adminCompany->update(['status' => 2]);

            DB::commit();
            return $this->successResponse('Administrador eliminado satisfactoriamente.', $adminCompany);
        } catch (\Throwable $th) {
            DB::rollBack();
            return $this->externalError('durante la eliminación de un jefe.', $th->getMessage());
        }
    }
}
