<?php

namespace App\Models;

use App\Observers\SalesmanObserver;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Salesman extends Model
{
    use HasFactory;
    protected $table = "tbl_salesman";

    protected $fillable = [
        'rut',
        'name',
        'surname',
        'gender',
        'type_person',
        'age',
        'birthdate',
        'address',
        'email',
        'phone',
        'idchannel',
        'idcompany',
        'status',
    ];

    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    public function channel(): HasOne
    {
        return $this->hasOne(SaleChannel::class, 'id', 'idchannel');
    }

    public function company(): HasOne
    {
        return $this->hasOne(Company::class, 'id', 'idcompany');
    }

    public function branchOffice(): HasMany
    {
        return $this->hasMany(SalesmanBranchOffice::class, 'idsalesman', 'id')->where('status', 1);
    }

    #Observer del modelo
    protected static function boot()
    {
        parent::boot();
        Salesman::observe(SalesmanObserver::class);
    }

    # Query scopes
    public function scopeActiveForID($query, $id)
    {
        return $query->where('id', $id)->where('status', 1)->company();
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeCompany($query)
    {
        return $query->when(
            request('idcompany'),
            fn ($query) => $query->where('idcompany', request('idcompany'))
        );
    }

    public function scopeRut($query, $rut, $id = null)
    {
        return $query->when($id, fn ($query) => $query->where('id', '<>', $id))->where('rut', $rut)->company();
    }

    # Filtros
    public function scopeSalesmanFilters($query)
    {
        #Filtro de Buscador
        $query->when(
            request('search'),
            function ($query) {
                $search = request('search');
                $query->where(function ($q) use ($search) {
                    $q->where('rut', 'LIKE', '%' . $search . '%')
                        ->orWhere('name', 'LIKE', '%' . $search . '%')
                        ->orWhere('surname', 'LIKE', '%' . $search . '%');
                });
            }
        );

        #Filtro de RUT
        $query->when(
            request('rut'),
            fn ($query) => $query->where('rut', 'LIKE', '%' . request('rut') . '%')
        );

        #Filtro de nombre
        $query->when(
            request('name'),
            fn ($query) => $query->where('name', 'LIKE', '%' . request('name') . '%')
        );

        #Filtro de apellidos
        $query->when(
            request('surname'),
            fn ($query) => $query->where('surname', 'LIKE', '%' . request('surname') . '%')
        );

        #Filtro de genero
        $query->when(
            request('gender'),
            fn ($query) => $query->where('gender', request('gender'))
        );

        #Filtro de tipo de persona
        $query->when(
            request('type_person'),
            fn ($query) => $query->where('type_person', request('type_person'))
        );

        #Filtro de tipo de edad
        $query->when(
            request('age'),
            fn ($query) => $query->where('age', request('age'))
        );

        #Filtro de tipo de nacimiento
        $query->when(
            request('birthdate'),
            fn ($query) => $query->where('birthdate', request('birthdate'))
        );

        #Filtro de tipo de direccion
        $query->when(
            request('address'),
            fn ($query) => $query->where('address', 'LIKE', '%' . request('address') . '%')
        );

        #Filtro de tipo de email
        $query->when(
            request('email'),
            fn ($query) => $query->where('email', 'LIKE', '%' . request('email') . '%')
        );

        #Filtro de tipo de teléfono
        $query->when(
            request('phone'),
            fn ($query) => $query->where('phone', 'LIKE', '%' . request('phone') . '%')
        );

        #Filtro de tipo de canal
        $query->when(
            request('idchannel'),
            fn ($query) => $query->where('idchannel', request('idchannel'))
        );

        #Filtro de idcompany
        $query->company();

        #Filtro de estados
        $query->when(
            request('status'),
            fn ($query) => $query->where('status', request('status'))
        )->when(
            !request('status'),
            fn ($query) => $query->where('status', 1)
        );

    }

    public function setUsername($username)
    {
        $this->username = $username;
    }
}
