<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;

class ProofPayment extends Model
{
    use HasFactory;
    protected $table = 'tbl_proof_payment';

    protected $fillable = [
        'idcompany',
        'payment_method',
        'idway_to_pay_for_company',
        'idstickets',
        'iduser_created',
        'date_admission',
        'document_date',
        'idbank',
        'serie',
        'amount',
        'amount_text',
        'unpaid_documents',
        'date_creation',
        'authorized_quota',
        'used_quota',
        'available_space',
        'expiration',
        'expiration_email',
        'document_url',
        'accepted_iduser',
        'status'
    ];

    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    protected $casts = [
        'idstickets'    => 'array'
    ];

    #Relaciones
    public function company(): HasOne
    {
        return $this->hasOne(Company::class, 'id', 'idcompany');
    }

    public function wayToPayForCompany(): HasOne
    {
        return $this->hasOne(WayToPayForCompany::class, 'id', 'idway_to_pay_for_company');
    }

    public function userCreated(): HasOne
    {
        return $this->hasOne(User::class, 'id', 'iduser_created');
    }

    public function acceptedUser(): HasOne
    {
        return $this->hasOne(User::class, 'id', 'accepted_iduser');
    }

    # Query scopes
    public function scopeActiveForID($query, $id)
    {
        return $query->where('id', $id)->where('status', 1);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeCompany($query)
    {
        return $query->when(
            request('idcompany'),
            fn ($query) => $query->where('idcompany', request('idcompany'))
        );
    }

    # Filtros
    public function scopeProofPaymentFilters($query)
    {
        #Filtro de id
        $query->when(
            request('id'),
            fn ($query) => $query->where('id', request('id'))
        );

        #Filtro de aceptados
        $query->when(request('accepted'), function ($query) {
            $query->where('payment_method', 1)
                  ->when(request('accepted') == 1, function ($query) {
                      // Cuando accepted es 1, accepted_iduser no debe ser nulo
                      $query->whereNotNull('accepted_iduser');
                  })
                  ->when(request('accepted') == 2, function ($query) {
                      // Cuando accepted es 2, accepted_iduser debe ser nulo
                      $query->whereNull('accepted_iduser');
                  });
        });

        #Filtro de métodos de pago
        $query->when(
            request('payment_method'),
            fn ($query) => $query->where('payment_method', request('payment_method'))
        );

        #Filtro de métodos de pago por compañía
        $query->when(
            request('idway_to_pay_for_company'),
            fn ($query) => $query->where('idway_to_pay_for_company', request('idway_to_pay_for_company'))
        );

        #Filtro de usuarios
        $query->when(
            request('iduser_created'),
            fn ($query) => $query->where('iduser_created', request('iduser_created'))
        );

        #Filtro de bancos
        $query->when(
            request('idbank'),
            fn ($query) => $query->where('idbank', request('idbank'))
        );

        #Filtro de correo enviado
        $query->when(
            request('expiration_email'),
            fn ($query) => $query->where('expiration_email', request('expiration_email'))
        );

        #Filtro de fecha
        $query->when(
            request('initial_date') && request('final_date'),
            function ($query) {
                $query->whereBetween('date_creation', [request('initial_date'), request('final_date')]);
            }
        );

        #Filtro de estados
        $query->when(
            request('status'),
            fn ($query) => $query->where('status', request('status'))
        )->when(
            !request('status'),
            fn ($query) => $query->where('status', 1)
        );
    }
}
