<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class ProductAttribute extends Model
{
    use HasFactory;
    protected $table = "tbl_product_attributes";
    protected $primaryKey = "id";

    protected $fillable = [
        'code',
        'name',
        'description',
        'idspecificattributes',
        'idtype',
        'idreference',
        'createdby',
        'idcompany',
        'status'
    ];

    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    protected $casts = [
        'idspecificattributes' => 'array'
    ];

    public function user(): HasOne
    {
        return $this->hasOne(User::class, 'id', 'createdby');
    }

    public function company(): HasOne
    {
        return $this->hasOne(Company::class, 'id', 'idcompany');
    }

    public function type(): HasOne
    {
        return $this->hasOne(TypeAttribute::class, 'id', 'idtype');
    }

    public function parentAttributes()
    {
        return $this->belongsToMany(
            ProductAttribute::class,         // Modelo relacionado
            'tbl_relation_product_attribute', // Tabla intermedia
            'child_attribute_id',            // Columna en la tabla intermedia que hace referencia al hijo
            'parent_attribute_id'            // Columna en la tabla intermedia que hace referencia al padre
        );
    }

    public function childAttributes()
    {
        return $this->belongsToMany(
            ProductAttribute::class,         // Modelo relacionado
            'tbl_relation_product_attribute', // Tabla intermedia
            'parent_attribute_id',           // Columna en la tabla intermedia que hace referencia al padre
            'child_attribute_id'             // Columna en la tabla intermedia que hace referencia al hijo
        );
    }

    public function subcategory()
    {
        return $this->belongsTo(Subcategory::class, 'idreference', 'id');
    }


    # Query scopes
    public function scopeActiveForID($query, $id)
    {
        return $query->where('id', $id)->where('status', 1)->company();
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeCompany($query)
    {
        return $query->when(
            request('idcompany'),
            fn ($query) => $query->where('idcompany', request('idcompany'))
        );
    }

    public function scopeCode($query, $code, $id = null)
    {
        return $query->when($id, fn ($query) => $query->where('id', '<>', $id))->where('code', $code);
    }

    public function scopeSpecificAtributtes($query, $idspecificattributes)
    {
        return $query->whereRaw("JSON_SEARCH(idspecificattributes, 'one', '$idspecificattributes') IS NOT NULL");
    }

    # Filtros
    public function scopeAttributeFilters($query)
    {
        #Filtro de Buscador
        $query->when(
            request('search'),
            fn ($query) => $query->where('code', 'LIKE', '%' . request('search') . '%')
                ->orWhere('name', 'LIKE', '%' . request('search') . '%')
        );

        #Filtro de codigo
        $query->when(
            request('code'),
            fn ($query) => $query->where('code', 'LIKE', '%' . request('code') . '%')
        );

        #Filtro de nombre
        $query->when(
            request('name'),
            fn ($query) => $query->where('name', 'LIKE', '%' . request('name') . '%')
        );

        #Filtro de descripcion
        $query->when(
            request('description'),
            fn ($query) => $query->where('description', 'LIKE', '%' . request('description') . '%')
        );

        #Filtro de atributos especificos
        $query->when(
            request('idspecificattribute'),
            fn ($query) => $query->whereRaw("JSON_SEARCH(idspecificattributes, 'one', '" . request('idspecificattribute') . "') IS NOT NULL")

        );

        #Filtro de id de usuario creador
        $query->when(
            request('createdby'),
            fn ($query) => $query->where('createdby', request('createdby'))
        );

        #Filtro de idcompany
        $query->company();

        #Filtro de estados
        $query->when(
            request('status') !== null,
            fn ($query) => $query->where('status', request('status'))
        )->when(
            request('status') === null,
            fn ($query) => $query->where('status', 1)
        );

    }
}
