<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;

class Product extends Model
{
    use HasFactory;
    protected $table = "tbl_products";
    protected $primaryKey = "id";

    protected $fillable = [
        'code_erp',
        'code_system',
        'barcode',
        'name',
        'description',
        'cost',
        'price',
        'alternativeprice',
        'margin',
        'offerprice',
        'minimumstock',
        'maximumstock',

        'main_image',
        'optional_image1',
        'optional_image2',
        'optional_image3',
        'optional_image4',

        'idattributes',
        'idspecificattributes',
        'iduses',
        'idspecificuses',

        'idcointype',
        'idcategory',
        'idsubcategory',
        'idspecificsubcategory',
        'idtype',
        'idsupplier',
        'idunitmeasure',
        'idunitmeasureoptional',
        'idbusinessarea',
        'idclassification',

        'idcompany',
        'status',
    ];

    protected $hidden = [
        'created_at',
        'updated_at'
    ];

    protected $casts = [
        'idattributes' => 'array',
        'idspecificattributes' => 'array',
        'iduses' => 'array',
        'idspecificuses' => 'array'
    ];

    # Relaciones

    public function relationProductAttributes()
    {
        return $this->belongsToMany(RelationProductAttribute::class, 'tbl_product_detail', 'product_id', 'relation_product_attribute_id')
                    ->withTimestamps();
    }

    public function cointType(): HasOne
    {
        return $this->hasOne(CoinType::class, 'id', 'idcointype');
    }

    public function category()
    {
        return $this->belongsTo(Category::class, 'idcategory');
    }
    
    public function subCategory()
    {
        return $this->belongsTo(Subcategory::class, 'idsubcategory');
    }
    
    public function specificSubCategory()
    {
        return $this->belongsTo(SpecificSubcategory::class, 'idspecificsubcategory');
    }
    

    public function type(): HasOne
    {
        return $this->hasOne(TypeProduct::class, 'id', 'idtype');
    }

    public function supplier(): HasOne
    {
        return $this->hasOne(Supplier::class, 'id', 'idsupplier');
    }

    public function unitMeasure(): HasOne
    {
        return $this->hasOne(UnitMeasure::class, 'id', 'idunitmeasure');
    }

    public function unitMeasureOptional(): HasOne
    {
        return $this->hasOne(UnitMeasure::class, 'id', 'idunitmeasureoptional');
    }

    public function businessArea(): HasOne
    {
        return $this->hasOne(BusinessArea::class, 'id', 'idbusinessarea');
    }

    public function classification(): HasOne
    {
        return $this->hasOne(ProductClassification::class, 'id', 'idclassification');
    }

    public function company(): HasOne
    {
        return $this->hasOne(Company::class, 'id', 'idcompany');
    }

    public function stock(): HasMany
    {
        return $this->hasMany(StockProduct::class, 'idproduct', 'id')->where('status', 1);
    }

    public function detailPriceListProduct(): HasMany
    {
        return $this->hasMany(DetailPriceListProduct::class, 'idproduct', 'id');
    }

    # Query scopes
    public function scopeActiveForID($query, $id)
    {
        return $query->where('id', $id)->where('status', 1)->company();
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeCompany($query)
    {
        return $query->when(
            request('idcompany'),
            fn ($query) => $query->where('idcompany', request('idcompany'))
        );
    }

    public function scopeCode($query, $code, $field, $id = null)
    {
        return $query->when($id, fn ($query) => $query->where('id', '<>', $id))->where($field, $code)->company();
    }

    public function scopeAtributtes($query, $idattributes)
    {
        return $query->whereRaw("JSON_SEARCH(idattributes, 'one', '$idattributes') IS NOT NULL");
    }

    public function scopeUses($query, $iduses)
    {
        return $query->whereRaw("JSON_SEARCH(iduses, 'one', '$iduses') IS NOT NULL");
    }

    public function scopeSpecificUses($query, $idspecificuses)
    {
        return $query->whereRaw("JSON_SEARCH(idspecificuses, 'one', '$idspecificuses') IS NOT NULL");
    }


    # Filtros
    public function scopeProductFilters($query)
    {
       # Filtro de Buscador
       $query->when(
        request('search'),
        function ($query) {
            $search = request('search');

            $query->where(function ($query) use ($search) {
                $query->where('code_system', 'LIKE', '%' . $search . '%')
                    ->orWhere('code_erp', 'LIKE', '%' . $search . '%')
                    ->orWhere('barcode', 'LIKE', '%' . $search . '%')
                    ->orWhere('name', 'LIKE', '%' . $search . '%')
                    ->orWhere('description', 'LIKE', '%' . $search . '%')
                    ->orWhereHas('specificSubCategory', function ($q) use ($search) {
                        $q->where('name', 'LIKE', '%' . $search . '%');
                    })
                    ->orWhereHas('relationProductAttributes', function ($q) use ($search) {
                        $q->whereHas('childAttribute', function ($childQuery) use ($search) {
                            $childQuery->where('name', 'LIKE', '%' . $search . '%');
                        });
                    })
                    ->orWhereHas('relationProductAttributes', function ($q) use ($search) {
                        $q->whereHas('parentAttribute', function ($parentQuery) use ($search) {
                            $parentQuery->where('name', 'LIKE', '%' . $search . '%');
                        });
                    });
            });
        }
    );

         # Filtro de modelo basado en el campo name de ProductAttribute
         $query->when(
            request('modelo'),
            function ($query, $modelo) {
                $query->whereHas('relationProductAttributes', function ($q) use ($modelo) {
                    $q->whereHas('childAttribute', function ($childQuery) use ($modelo) {
                        $childQuery->where('name', 'LIKE', '%' . $modelo . '%');
                    });
                });
            }
        );

        $query->when(
            request('marca'),
            function ($query, $marca) {
                $query->whereHas('relationProductAttributes', function ($q) use ($marca) {
                    $q->whereHas('parentAttribute', function ($parentQuery) use ($marca) {
                        $parentQuery->where('name', 'LIKE', '%' . $marca . '%');
                    });
                });
            }
        );
        

       # Filtro de código general para ambas columnas
        $query->when(
            request('code'),
            fn ($query) => $query->where(function ($query) {
                $code = request('code');
                $query->where('code_system', 'LIKE', '%' . $code . '%')
                    ->orWhere('code_erp', 'LIKE', '%' . $code . '%');
            })
        );


        #Filtro de barras
        $query->when(
            request('barcode'),
            fn ($query) => $query->where('barcode', 'LIKE', '%' . request('barcode') . '%')
        );

        #Filtro de nombre
        // $query->when(
        //     request('name'),
        //     fn ($query) => $query->where('name', 'LIKE', '%' . request('name') . '%')
        // );

        #Filtro de atributos
        $query->when(
            request('idattribute'),
            fn ($query) => $query->whereRaw("JSON_SEARCH(idattributes, 'one', '" . request('idattribute') . "') IS NOT NULL")

        );

        #Filtro de atributos especificos
        $query->when(
            request('idspecificattribute') || request('idspecific_attribute'),
            fn ($query) => $query->whereRaw("JSON_SEARCH(idspecificattributes, 'one', '" . request('idspecificattribute') ??  request('idspecific_attribute') . "') IS NOT NULL")

        );

        #Filtro de tipo de moneda
        $query->when(
            request('idcointype'),
            fn ($query) => $query->where('idcointype', request('idcointype'))

        );

        #Filtro de categoria
        $query->when(
            request('idcategory') || request('idcategory_product'),
            fn ($query) => $query->where('idcategory', request('idcategory') ?? request('idcategory_product'))

        );

        #Filtro de nombre categoria
        $query->when(
            request('category_name'),
            function ($query, $category) {
                $query->whereHas('category', function ($q) use ($category) {
                    $q->where('name', 'LIKE', '%' . $category . '%');
                });
            }
        ); 

        #Filtro de subcategoria
        $query->when(
            request('idsubcategory') || request('idsubcategory_product'),
            fn ($query) => $query->where('idsubcategory', request('idsubcategory') ?? request('idsubcategory_product'))
        );

        #Filtro de nombre subcategoria
        $query->when(
            request('subcategory_name'),
            function ($query, $subCategory) {
                $query->whereHas('subCategory', function ($q) use ($subCategory) {
                    $q->where('name', 'LIKE', '%' . $subCategory . '%');
                });
            }
        ); 

        #Filtro de subcategoria especifica
        $query->when(
            request('idspecificsubcategory'),
            fn ($query) => $query->where('idspecificsubcategory', request('idspecificsubcategory'))
        );

         #Filtro de nombre subcategoria especifica
         $query->when(
            request('line'),
            function ($query, $specificSubCategory) {
                $query->whereHas('specificSubCategory', function ($q) use ($specificSubCategory) {
                    $q->where('name', 'LIKE', '%' . $specificSubCategory . '%');
                });
            }
        ); 

        #Filtro de tipo
        $query->when(
            request('idtype'),
            fn ($query) => $query->where('idtype', request('idtype'))
        );

        #Filtro de proveedor
        $query->when(
            request('idsupplier'),
            fn ($query) => $query->where('idsupplier', request('idsupplier'))
        );

        #Filtro de unidad de medida
        $query->when(
            request('idunitmeasure'),
            fn ($query) => $query->where('idunitmeasure', request('idunitmeasure'))
        );

        #Filtro de unidad de medida opcional
        $query->when(
            request('idunitmeasureoptional'),
            fn ($query) => $query->where('idunitmeasureoptional', request('idunitmeasureoptional'))
        );

        #Filtro de area de negocio
        $query->when(
            request('idbusinessarea'),
            fn ($query) => $query->where('idbusinessarea', request('idbusinessarea'))
        );

        #Filtro de clasificación
        $query->when(
            request('idclassification'),
            fn ($query) => $query->where('idclassification', request('idclassification'))
        );

        #Filtro de sucursal
        $query->when(
            request('idbranchoffice'),
            fn ($query) => $query->whereHas(
                'businessArea',
                fn ($q) => $q->where('idbranchoffice', request('idbranchoffice'))
            )
        );

        #Filtro de distrito
        $query->when(
            request('iddistrict'),
            fn ($query) => $query->whereHas(
                'businessArea.branchOffice',
                fn ($q) => $q->where('iddistrict', request('iddistrict'))
            )
        );

        #Filtro de provincia
        $query->when(
            request('idprovince'),
            fn ($query) => $query->whereHas(
                'businessArea.branchOffice',
                fn ($q) => $q->where('idprovince', request('idprovince'))
            )
        );

        #Filtro de departamento
        $query->when(
            request('iddepartment'),
            fn ($query) => $query->whereHas(
                'businessArea.branchOffice',
                fn ($q) => $q->where('iddepartment', request('iddepartment'))
            )
        );

        #Filtro de pais
        $query->when(
            request('idcountry'),
            fn ($query) => $query->whereHas(
                'businessArea.branchOffice',
                fn ($q) => $q->where('idcountry', request('idcountry'))
            )
        );

        #Filtro de idcompany
        $query->company();

        #Filtro de estados
        $query->when(
            request('status') !== null,
            fn ($query) => $query->where('status', request('status'))
        )->when(
            request('status') === null,
            fn ($query) => $query->where('status', 1)
        );
    }
}
