<?php

namespace App\Models;

use App\Observers\CompanyObserver;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\HasOne;

class Company extends Model
{
    use HasFactory;

    protected $table = 'tbl_company';
    protected $primaryKey = 'id';

    protected $fillable = [
        'rut',
        'business_name',
        'address',
        'email',
        'phone',
        'giro',
        'iddistrict',
        'code_comerce_wp',
        'api_key_wp',
        'idmembership',
        'status'
    ];

    protected $hidden = [
        'created_at',
        'updated_at',
        'code_comerce_wp',
        'api_key_wp'
    ];

    #Observer del modelo
    protected static function boot()
    {
        parent::boot();
        Company::observe(CompanyObserver::class);
    }

    # Relaciones
    public function district(): HasOne
    {
        return $this->hasOne(District::class, 'id', 'iddistrict');
    }

    public function membership(): HasOne
    {
        return $this->hasOne(Membership::class, 'id', 'idmembership');
    }

    public function companyConfiguration(): HasOne
    {
        return $this->hasOne(CompanyConfiguration::class, 'idcompany', 'id');
    }

    # Query scopes
    public function scopeActiveForID($query, $id)
    {
        return $query->where('id', $id)->where('status', 1);
    }

    public function scopeActive($query)
    {
        return $query->where('status', 1);
    }

    public function scopeRut($query, $rut, $id = null)
    {
        return $query->when($id, fn ($query) => $query->where('id', '<>', $id))->where('rut', $rut);
    }

    # Filtros
    public function scopeCompanyFilters($query)
    {
        #Filtro de Buscador
        $query->when(
            request('search'),
            fn ($query) => $query->where('rut', 'LIKE', '%' . request('search') . '%')
                ->orWhere('business_name', 'LIKE', '%' . request('search') . '%')
        );

        #Filtro de RUT
        $query->when(
            request('rut'),
            fn ($query) => $query->where('rut', 'LIKE', '%' . request('rut') . '%')
        );

        #Filtro de nombre
        $query->when(
            request('business_name'),
            fn ($query) => $query->where('business_name', 'LIKE', '%' . request('business_name') . '%')
        );

        #Filtro de estados
        $query->when(
            request('status'),
            fn ($query) => $query->where('status', request('status'))
        )->when(
            !request('status'),
            fn ($query) => $query->where('status', 1)
        );

    }
}
